<?php


if (!defined('ABSPATH')) {
    exit; // Evita el acceso directo
}

// FUNCIÓN PARA CALCULAR PUNTOS Y GUARDARLOS EN LA TABLA mnkt_registro_horario
function calcular_puntos_gamificacion($user_id, $registro_id) {
    global $wpdb;
    $tabla_registro = $wpdb->prefix . 'registro_horario';

    // Obtener el registro de entrada y salida
    $registro = $wpdb->get_row(
        $wpdb->prepare(
            "SELECT entrada, salida, TIMEDIFF(salida, entrada) AS duracion 
            FROM $tabla_registro 
            WHERE id = %d AND user_id = %d AND salida IS NOT NULL",
            $registro_id, $user_id
        )
    );

    if (!$registro) {
        return; // No hay registro válido
    }

    $total_puntos = 0;
    $duracion = strtotime($registro->duracion) - strtotime('00:00:00'); // Convertir a segundos

    // ✅ Recompensa por ticaje válido (más granular)
    if ($duracion >= 1.5 * 3600 && $duracion < 2 * 3600) {
        $total_puntos += 20; // Sesión entre 1.5h y 2h
    } elseif ($duracion >= 2 * 3600 && $duracion < 3 * 3600) {
        $total_puntos += 45; // Sesión entre 2h y 3h
    } elseif ($duracion >= 3 * 3600) {
        $total_puntos += 60; // Sesión de más de 3h
    } else {
        $total_puntos -= 30; // Penalización por sesiones muy cortas
    }

    // ✅ Bonus por descansos correctos con fraccionamiento
    $descanso_puntos = [ // Diferentes tipos de descansos con variabilidad en los puntos
        'corto' => [[25 * 60, 30 * 60, 18], [30 * 60, 35 * 60, 22]], // 18-22 puntos según duración
        'medio' => [[45 * 60, 55 * 60, 30]], // Descanso intermedio
        'largo' => [[90 * 60, 120 * 60, 48], [120 * 60, 135 * 60, 55]] // 48-55 puntos según duración
    ];

    $descansos = []; // Array para almacenar descansos

    // Simulación de descansos basada en registros anteriores
    if ($registro) {
        $ultimo_registro = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT salida FROM $tabla_registro WHERE user_id = %d AND id < %d ORDER BY id DESC LIMIT 1",
                $user_id, $registro_id
            )
        );

        if ($ultimo_registro) {
            $descanso = strtotime($registro->entrada) - strtotime($ultimo_registro);
            $descansos[] = $descanso;
        }
    }

    foreach ($descansos as $descanso) {
        foreach ($descanso_puntos as $tipo => $rangos) {
            foreach ($rangos as $rango) {
                if ($descanso >= $rango[0] && $descanso <= $rango[1]) {
                    $total_puntos += $rango[2];
                }
            }
        }
    }

    // ✅ Bonus por jornada entre 7h y 8h con variabilidad
    if ($duracion >= 7 * 3600 && $duracion <= 8 * 3600) {
        $total_puntos += 100;

        // **Nuevo**: Bonus de 1 a 2 puntos cada 3-5 minutos dentro del rango correcto
        $puntos_adicionales = floor(($duracion - (7 * 3600)) / rand(180, 300)); // Entre 3 y 5 min
        $total_puntos += $puntos_adicionales;
    }

    // ❌ Penalización progresiva por exceder 8 horas
    if ($duracion > 8 * 3600) {
        $exceso = $duracion - (8 * 3600);
        $penalizacion = floor($exceso / rand(600, 900)) * rand(8, 12); // -8 a -12 puntos cada 10-15 min
        $total_puntos -= $penalizacion;
    }

    // Guardar los puntos en la tabla mnkt_registro_horario
    $wpdb->update(
        $tabla_registro,
        ['puntos' => $total_puntos],
        ['id' => $registro_id]
    );

    return $total_puntos;
}

// FUNCIÓN PARA EJECUTAR GAMIFICACIÓN CUANDO UN USUARIO REGISTRA SU SALIDA
function gamificacion_al_registrar_salida($user_id, $registro_id) {
    $puntos_ganados = calcular_puntos_gamificacion($user_id, $registro_id);
    error_log("Gamificación: Usuario $user_id ganó $puntos_ganados puntos en el registro $registro_id.");
}

// REGISTRAR EL HOOK CUANDO UN USUARIO MARCA SU SALIDA
function registrar_gamificacion_salida($user_id, $registro_id) {
    gamificacion_al_registrar_salida($user_id, $registro_id);
}

// AÑADIR EL HOOK PARA EJECUTAR AUTOMÁTICAMENTE AL CERRAR UNA SESIÓN
add_action('registrar_salida_trabajador', 'registrar_gamificacion_salida', 10, 2);
