<?php
/**
 * Plugin Name: Despapela - Alta Empresa + Gestor (Vue)
 * Description: Shortcode [despapela_alta_empresa] para crear empresa, crear/reutilizar Gestor y asignarlo. Front con Vue 3 CDN + AJAX seguro.
 * Version: 1.0.0
 * Author: Despapela
 */

if (!defined('ABSPATH')) exit;

class Despapela_Alta_Empresa {
    private $nonce_action = 'despapela_alta_empresa_nonce';
    private $ajax_action  = 'despapela_crear_empresa';

    public function __construct() {
        add_shortcode('despapela_alta_empresa', [$this, 'render_shortcode']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue']);
        add_action("wp_ajax_{$this->ajax_action}", [$this, 'handle_ajax']);
        add_action("wp_ajax_nopriv_{$this->ajax_action}", [$this, 'handle_ajax']); // si quieres permitir alta pública
    }

    public function enqueue() {
        // Vue 3 por CDN
        wp_register_script('vue3', 'https://unpkg.com/vue@3/dist/vue.global.prod.js', [], null, true);
        // Estilos mínimos
        $css = "
        .dp-card{max-width:860px;margin:24px auto;padding:24px;border:1px solid #e5e7eb;border-radius:12px;background:#fff;box-shadow:0 4px 14px rgba(0,0,0,.06)}
        .dp-grid{display:grid;gap:14px;grid-template-columns:1fr 1fr}
        .dp-grid-1{grid-template-columns:1fr}
        .dp-field{display:flex;flex-direction:column;gap:6px}
        .dp-field label{font-weight:600;color:#0f172a}
        .dp-field input,.dp-field select,.dp-field textarea{border:1px solid #cbd5e1;border-radius:8px;padding:10px 12px;font-size:14px}
        .dp-2{grid-column:span 2}
        .dp-actions{display:flex;gap:10px;justify-content:flex-end;margin-top:10px}
        .dp-btn{background:#005B73;color:#fff;border:none;border-radius:10px;padding:10px 14px;font-weight:600;cursor:pointer}
        .dp-btn:disabled{opacity:.6;cursor:not-allowed}
        .dp-muted{color:#475569;font-size:13px}
        .dp-error{color:#b91c1c;font-size:13px}
        .dp-ok{background:#e6fffa;border:1px solid #99f6e4;color:#065f46;padding:10px;border-radius:8px}
        .dp-warn{background:#fff7ed;border:1px solid #fed7aa;color:#9a3412;padding:10px;border-radius:8px}
        ";
        wp_register_style('despapela-alta-empresa', false, [], null);
        wp_add_inline_style('despapela-alta-empresa', $css);
    }

    public function render_shortcode() {
        wp_enqueue_script('vue3');
        wp_enqueue_style('despapela-alta-empresa');

        $nonce = wp_create_nonce($this->nonce_action);

        ob_start(); ?>
        <div class="dp-card" id="dp-alta-empresa"
             data-ajax-url="<?php echo esc_url(admin_url('admin-ajax.php')); ?>"
             data-ajax-action="<?php echo esc_attr($this->ajax_action); ?>"
             data-nonce="<?php echo esc_attr($nonce); ?>">
          <h2 style="margin-top:0">Alta de Empresa + Asignación de Gestor</h2>
          <p class="dp-muted">Crea la empresa, da de alta al gestor (o reutiliza si existe) y lo asigna automáticamente.</p>

          <div v-if="ui.success" class="dp-ok">{{ ui.success }}</div>
          <div v-if="ui.warning" class="dp-warn">{{ ui.warning }}</div>
          <div v-if="ui.error" class="dp-error" style="margin-bottom:8px">{{ ui.error }}</div>

          <form @submit.prevent="submit" novalidate>
            <h3>Datos de la Empresa</h3>
            <div class="dp-grid">
              <div class="dp-field">
                <label>CIF <span style="color:#b91c1c">*</span></label>
                <input v-model.trim="empresa.user_cif" placeholder="B12345678" required>
                <small class="dp-error" v-if="errors.user_cif">{{ errors.user_cif }}</small>
              </div>

              <div class="dp-field">
                <label>Razón social <span style="color:#b91c1c">*</span></label>
                <input v-model.trim="empresa.razon_social" placeholder="Tu Empresa S.L." required>
                <small class="dp-error" v-if="errors.razon_social">{{ errors.razon_social }}</small>
              </div>

              <div class="dp-field dp-2">
                <label>Dirección de facturación</label>
                <textarea v-model.trim="empresa.direccion_facturacion" rows="2" placeholder="C/ Ejemplo, 1 · 30001 Murcia"></textarea>
              </div>

              <div class="dp-field">
                <label>Método de pago</label>
                <select v-model="empresa.metodo_pago">
                  <option value="">Seleccionar…</option>
                  <option>Domiciliación</option>
                  <option>Tarjeta</option>
                  <option>Transferencia</option>
                </select>
              </div>

              <div class="dp-field">
                <label>Gamificación</label>
                <select v-model="empresa.gamificacion">
                  <option value="desactivado">Desactivado</option>
                  <option value="activado">Activado</option>
                </select>
              </div>

              <div class="dp-field dp-2">
                <label>
                  <input type="checkbox" v-model="empresa.aceptacion_proteccion_datos"> Aceptación Protección de Datos
                </label>
                <label>
                  <input type="checkbox" v-model="empresa.aceptacion_sepa"> Aceptación SEPA
                </label>
              </div>
            </div>

            <h3 style="margin-top:18px">Datos del Gestor</h3>
            <div class="dp-grid">
              <div class="dp-field">
                <label>Nombre a mostrar</label>
                <input v-model.trim="gestor.display_name" placeholder="Nombre y apellidos">
              </div>
              <div class="dp-field">
                <label>Email <span style="color:#b91c1c">*</span></label>
                <input v-model.trim="gestor.user_email" type="email" placeholder="gestor@empresa.com" required>
                <small class="dp-error" v-if="errors.user_email">{{ errors.user_email }}</small>
              </div>
              <div class="dp-field">
                <label>DNI</label>
                <input v-model.trim="gestor.user_dni" placeholder="00000000A">
              </div>
            </div>

            <div class="dp-actions">
              <button type="button" class="dp-btn" @click="reset" :disabled="ui.loading">Limpiar</button>
              <button type="submit" class="dp-btn" :disabled="ui.loading">
                {{ ui.loading ? 'Creando…' : 'Crear empresa y asignar gestor' }}
              </button>
            </div>
          </form>
        </div>

        <script>
        (() => {
          const { createApp, reactive } = Vue;
          createApp({
            setup() {
              const el = document.getElementById('dp-alta-empresa');
              const ajaxUrl = el.dataset.ajaxUrl;
              const action  = el.dataset.ajaxAction;
              const nonce   = el.dataset.nonce;

              const empresa = reactive({
                user_cif: '',
                razon_social: '',
                direccion_facturacion: '',
                metodo_pago: '',
                gamificacion: 'desactivado',
                aceptacion_proteccion_datos: false,
                aceptacion_sepa: false,
              });

              const gestor = reactive({
                display_name: '',
                user_email: '',
                user_dni: '',
              });

              const ui = reactive({ loading:false, error:'', success:'', warning:'' });
              const errors = reactive({});

              function validate() {
                ui.error=''; ui.success=''; ui.warning=''; Object.keys(errors).forEach(k=>delete errors[k]);

                if (!empresa.user_cif) errors.user_cif = 'El CIF es obligatorio.';
                if (!empresa.razon_social) errors.razon_social = 'La razón social es obligatoria.';
                if (!gestor.user_email) errors.user_email = 'El email del gestor es obligatorio.';
                if (gestor.user_email && !/^\S+@\S+\.\S+$/.test(gestor.user_email)) errors.user_email = 'Email no válido.';
                return Object.keys(errors).length === 0;
              }

              async function submit() {
                if (!validate()) return;
                ui.loading = true;
                try {
                  const form = new FormData();
                  form.append('action', action);
                  form.append('_wpnonce', nonce);
                  form.append('payload', JSON.stringify({ empresa, gestor }));

                  const res = await fetch(ajaxUrl, { method:'POST', body: form, credentials:'same-origin' });
                  const data = await res.json();
                  if (!res.ok || !data?.success) throw new Error(data?.message || 'Error desconocido');

                  ui.success = 'Empresa y gestor creados/asignados correctamente.';
                  if (data.info?.password_plain) {
                    ui.warning = `Gestor nuevo creado. Guarda esta contraseña: ${data.info.password_plain}`;
                  }
                  // opcional: reset
                  // reset();
                } catch (e) {
                  ui.error = e.message || 'Error al procesar la solicitud.';
                } finally {
                  ui.loading = false;
                }
              }

              function reset() {
                empresa.user_cif=''; empresa.razon_social=''; empresa.direccion_facturacion='';
                empresa.metodo_pago=''; empresa.gamificacion='desactivado';
                empresa.aceptacion_proteccion_datos=false; empresa.aceptacion_sepa=false;
                gestor.display_name=''; gestor.user_email=''; gestor.user_dni='';
                ui.error=''; ui.success=''; ui.warning='';
                Object.keys(errors).forEach(k=>delete errors[k]);
              }

              return { empresa, gestor, ui, errors, submit, reset };
            }
          }).mount('#dp-alta-empresa');
        })();
        </script>
        <?php
        return ob_get_clean();
    }

    public function handle_ajax() {
        check_ajax_referer($this->nonce_action);

        $payload = isset($_POST['payload']) ? json_decode(stripslashes($_POST['payload']), true) : null;
        if (!$payload || empty($payload['empresa']) || empty($payload['gestor'])) {
            return $this->json(false, 'Payload inválido.');
        }

        global $wpdb;
        $t_empresas = $wpdb->prefix . 'empresas';
        $t_users    = $wpdb->prefix . 'users'; // OJO: tu tabla personalizada se llama mnkt_users, así que usamos prefix+'users' => mnkt_users
        $t_rel      = $wpdb->prefix . 'empresas_usuarios';

        $empresa = [
            'user_cif'                     => $this->sanitize_cif($payload['empresa']['user_cif'] ?? ''),
            'razon_social'                 => sanitize_text_field($payload['empresa']['razon_social'] ?? ''),
            'direccion_facturacion'        => sanitize_textarea_field($payload['empresa']['direccion_facturacion'] ?? ''),
            'metodo_pago'                  => sanitize_text_field($payload['empresa']['metodo_pago'] ?? ''),
            'aceptacion_proteccion_datos'  => !empty($payload['empresa']['aceptacion_proteccion_datos']) ? 1 : 0,
            'aceptacion_sepa'              => !empty($payload['empresa']['aceptacion_sepa']) ? 1 : 0,
            'gamificacion'                 => in_array(($payload['empresa']['gamificacion'] ?? 'desactivado'), ['activado','desactivado']) ? $payload['empresa']['gamificacion'] : 'desactivado',
            'estado_empresa'               => 'activado',
            'estado_validacion'            => 'Pendiente',
            'fecha_aceptacion'             => current_time('mysql'),
            'ip_firma'                     => $_SERVER['REMOTE_ADDR'] ?? '',
            'user_agent'                   => substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 255),
        ];

        $gestor = [
            'display_name' => sanitize_text_field($payload['gestor']['display_name'] ?? ''),
            'user_email'   => sanitize_email($payload['gestor']['user_email'] ?? ''),
            'user_dni'     => sanitize_text_field($payload['gestor']['user_dni'] ?? ''),
        ];

        if (!$empresa['user_cif'] || !$empresa['razon_social'] || !$gestor['user_email']) {
            return $this->json(false, 'Faltan campos obligatorios.');
        }

        // 1) Insertar/recuperar empresa por user_cif
        $empresa_id = $wpdb->get_var($wpdb->prepare("SELECT id FROM {$t_empresas} WHERE user_cif=%s", $empresa['user_cif']));
        if ($empresa_id) {
            // actualizar datos clave
            $wpdb->update($t_empresas, $empresa, ['id' => $empresa_id]);
        } else {
            $ok = $wpdb->insert($t_empresas, $empresa, [
                '%s','%s','%s','%s','%d','%d','%s','%s','%s','%s','%s'
            ]);
            if (!$ok) return $this->json(false, 'No se pudo crear la empresa.');
            $empresa_id = (int)$wpdb->insert_id;
        }

        // 2) Insertar/recuperar usuario gestor por email (en mnkt_users)
        // user_login basado en email
        $user_login = sanitize_user(current(explode('@', $gestor['user_email'])));
        if (!$user_login) $user_login = 'gestor_' . wp_generate_password(8, false);

        $user_id = $wpdb->get_var($wpdb->prepare("SELECT ID FROM {$t_users} WHERE user_email=%s", $gestor['user_email']));

        $info = [];
        if ($user_id) {
            // actualizar datos útiles (opcional)
            $wpdb->update($t_users, [
                'display_name' => $gestor['display_name'] ?: $wpdb->get_var($wpdb->prepare("SELECT display_name FROM {$t_users} WHERE ID=%d", $user_id)),
                'user_dni'     => $gestor['user_dni'],
                'user_cif'     => $empresa['user_cif'],
                'user_cif_id'  => $empresa_id,
                'user_rol'     => 'Gestor',
                'user_estado'  => 'Activado',
            ], ['ID' => $user_id]);
        } else {
            // crear usuario nuevo en mnkt_users (hash WP)
            $plain = wp_generate_password(10, false);
            $hash  = wp_hash_password($plain);

            $ok = $wpdb->insert($t_users, [
                'user_login'    => $user_login,
                'user_pass'     => $hash,
                'user_nicename' => sanitize_title($user_login),
                'user_email'    => $gestor['user_email'],
                'user_registered' => current_time('mysql'),
                'display_name'  => $gestor['display_name'] ?: $user_login,
                'user_dni'      => $gestor['user_dni'],
                'user_cif'      => $empresa['user_cif'],
                'user_cif_id'   => $empresa_id,
                'user_rol'      => 'Gestor',
                'user_estado'   => 'Activado',
            ], [
                '%s','%s','%s','%s','%s','%s','%s','%s','%d','%s','%s'
            ]);
            if (!$ok) return $this->json(false, 'No se pudo crear el usuario Gestor.');
            $user_id = (int)$wpdb->insert_id;
            $info['password_plain'] = $plain; // dev: puedes mostrarla/mandarla por email
        }

        // 3) Asignar relación en mnkt_empresas_usuarios (rol Gestor)
        // evitar duplicados por UNIQUE(user_id, empresa_id, rol_en_empresa)
        $ya = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$t_rel} WHERE user_id=%d AND empresa_id=%d AND rol_en_empresa='Gestor'",
            $user_id, $empresa_id
        ));
        if (!$ya) {
            $ok = $wpdb->insert($t_rel, [
                'user_id'       => $user_id,
                'empresa_id'    => $empresa_id,
                'rol_en_empresa'=> 'Gestor',
                'estado'        => 'Activado',
                'creado_en'     => current_time('mysql'),
            ], ['%d','%d','%s','%s','%s']);
            if (!$ok) return $this->json(false, 'No se pudo asignar el Gestor a la empresa.');
        }

        return $this->json(true, 'OK', [
            'empresa_id' => $empresa_id,
            'user_id'    => $user_id,
            'info'       => $info,
        ]);
    }

    private function sanitize_cif($cif) {
        $cif = strtoupper(preg_replace('/\s+/', '', $cif));
        // Validación ligera: 1 letra + 7 dígitos + 1 letra/dígito (muy laxa, pero suficiente para front)
        if (!preg_match('/^[A-Z]\d{7}[A-Z0-9]$/', $cif)) {
            // si no pasa, igualmente devuelve limpio para no bloquear pruebas
        }
        return $cif;
    }

    private function json($success, $message, $extra = []) {
        $resp = ['success' => (bool)$success, 'message' => $message];
        if (!empty($extra)) $resp = array_merge($resp, $extra);
        wp_send_json($resp, $success ? 200 : 400);
    }
}

new Despapela_Alta_Empresa();
