<?php
/**
 * Gestor unificado de estado de empresas (activar/desactivar)
 * Reglas:
 *  - Mientras el trial esté en curso (ahora < trial_expires_at): ACTIVA.
 *  - Tras el trial:
 *      - Si la suscripción en Stripe está ACTIVA => ACTIVA (trial_status=convertido).
 *      - En cualquier otro caso (cancelled, unpaid, past_due, sin sub): DESACTIVA (trial_status=expirado si procede).
 *
 * Integra:
 *  - Webhook Stripe (checkout.session.completed, customer.subscription.* , invoice.payment_failed)
 *  - Cron (normalización periódica)
 *  - Función pública para normalizar una empresa concreta
 *
 * Requisitos:
 *  - CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET en wp-config.php (webhook)
 *  - CONTROL_HORARIO_STRIPE_SECRET_KEY en wp-config.php o en la opción ch_stripe_secret_key (para consultas a Stripe)
 *  - SDK en /lib/stripe-php/init.php
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('CH_Estado_Empresas')) {

class CH_Estado_Empresas {
    const NS  = 'control-horario/v1';
    const RT  = 'stripe-webhook';
    const LOG = 'control-horario-estado.log';

    private $t_empresas = 'mnkt_empresas';
    private $t_rel      = 'mnkt_empresas_usuarios';
    private $t_users    = 'mnkt_users';

    public function __construct() {
        // SDK
        add_action('plugins_loaded', [$this, 'load_sdk'], 1);

        // Webhook (si quieres mantener un endpoint único aquí)
        add_action('rest_api_init', function () {
            // register_rest_route(self::NS, '/' . self::RT, [
            //     'methods'  => ['POST','GET'],
            //     'callback' => [$this, 'route'],
            //     'permission_callback' => '__return_true',
            // ]);
        });

        // Cron: normalización 2 veces al día
        add_action('init', [$this, 'schedule_cron']);
        add_action('ch_normalize_company_states', [$this, 'cron_normalize_all']);
    }

    public function load_sdk() {
        $base = defined('CH_STRIPE_WH_PLUGIN_DIR') ? CH_STRIPE_WH_PLUGIN_DIR : trailingslashit(dirname(__DIR__));
        $sdk  = $base . 'lib/stripe-php/init.php';
        if (file_exists($sdk)) require_once $sdk;
    }

    private function log($msg, $ctx = []) {
        $up = wp_get_upload_dir();
        @file_put_contents(
            trailingslashit($up['basedir']).self::LOG,
            '['.date('Y-m-d H:i:s').'] '.$msg.($ctx?' | '.json_encode($ctx,JSON_UNESCAPED_UNICODE):'').PHP_EOL,
            FILE_APPEND
        );
    }

    private function get_stripe_client() {
        if (!class_exists('\Stripe\StripeClient')) return null;
        $secret = defined('CONTROL_HORARIO_STRIPE_SECRET_KEY')
            ? CONTROL_HORARIO_STRIPE_SECRET_KEY
            : get_option('ch_stripe_secret_key', '');
        if (!$secret) return null;
        try { return new \Stripe\StripeClient($secret); } catch (\Throwable $e) { return null; }
    }

    /* ========== ROUTER / WEBHOOK ========== */
    public function route(\WP_REST_Request $req){
        if ($req->get_method()==='GET') {
            return new \WP_REST_Response(['ok'=>true],200);
        }
        if (!class_exists('\Stripe\Webhook')) {
            $this->log('Stripe SDK no cargado');
            return new \WP_REST_Response(['error'=>'Stripe SDK no cargado'],500);
        }
        if (!defined('CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET') || !CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET) {
            $this->log('Secret webhook no definido');
            return new \WP_REST_Response(['error'=>'Webhook secret no definido'],500);
        }

        $payload = $req->get_body();
        $sig     = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';

        try {
            $event = \Stripe\Webhook::constructEvent($payload, $sig, CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET);
        } catch (\Throwable $e) {
            $this->log('Webhook error', ['e'=>$e->getMessage()]);
            return new \WP_REST_Response(['error'=>'Invalid'],400);
        }

        $type = $event->type ?? '';
        $obj  = $event->data->object ?? null;
        $this->log('Webhook', ['type'=>$type]);

        try {
            switch ($type) {
                case 'checkout.session.completed':
                    $this->on_checkout_completed($obj);
                    break;

                case 'customer.subscription.created':
                case 'customer.subscription.updated':
                    $this->on_subscription_upd($obj);
                    break;

                case 'customer.subscription.deleted':
                case 'customer.subscription.canceled':
                    $this->on_subscription_cancel($obj);
                    break;

                case 'invoice.payment_failed':
                    $this->on_invoice_failed($obj);
                    break;

                default:
                    // ignorar
                    break;
            }
        } catch (\Throwable $e) {
            $this->log('Webhook handler error', ['e'=>$e->getMessage()]);
        }

        return new \WP_REST_Response(['ok'=>true],200);
    }

    /* ========== REGLAS DE ESTADO ========== */

    /**
     * Normaliza una empresa concreta por ID.
     * - Lee trial y fechas
     * - Consulta Stripe si hay subscription_id o al menos customer_id
     * - Actualiza trial_status / estado_empresa
     */
    public function normalize_empresa($empresa_id) {
        global $wpdb;
        $e = $wpdb->get_row($wpdb->prepare("
            SELECT id, user_cif, razon_social, estado_empresa,
                   trial_status, trial_expires_at,
                   stripe_customer_id, stripe_subscription_id
            FROM {$this->t_empresas}
            WHERE id=%d
        ", $empresa_id));
        if (!$e) return false;

        $now    = current_time('timestamp');
        $exp_ts = $e->trial_expires_at ? strtotime(get_date_from_gmt($e->trial_expires_at)) : 0;

        // 1) Si el trial aún sigue en curso -> ACTIVA y no miramos Stripe
        if ($e->trial_status === 'en_curso' && $exp_ts > $now) {
            if ($e->estado_empresa !== 'activado') {
                $wpdb->update($this->t_empresas, ['estado_empresa'=>'activado'], ['id'=>$e->id]);
            }
            return true;
        }

        // 2) Trial terminó: consultar Stripe
        $stripe_status = $this->resolve_subscription_status($e->stripe_subscription_id, $e->stripe_customer_id);

        if ($stripe_status === 'active') {
            // convertido + activado
            $wpdb->update($this->t_empresas, [
                'trial_status'   => 'convertido',
                'estado_empresa' => 'activado',
            ], ['id'=>$e->id]);
        } else {
            // expirado + desactivado
            $wpdb->update($this->t_empresas, [
                'trial_status'   => 'expirado',
                'estado_empresa' => 'desactivado',
                'deactivated_at' => current_time('mysql'),
            ], ['id'=>$e->id]);
        }

        return true;
    }

    /**
     * Devuelve 'active' si hay una sub activa. Si no, 'none' u otro status ('canceled','unpaid','past_due',...)
     */
    private function resolve_subscription_status($subscription_id, $customer_id) {
        $cli = $this->get_stripe_client();
        if (!$cli) return 'none';

        try {
            if (!empty($subscription_id)) {
                $sub = $cli->subscriptions->retrieve($subscription_id, []);
                return $sub->status ?? 'none';
            }
            if (!empty($customer_id)) {
                // Busca la más reciente
                $subs = $cli->subscriptions->all(['customer'=>$customer_id, 'limit'=>1]);
                if (!empty($subs->data)) {
                    return $subs->data[0]->status ?? 'none';
                }
            }
        } catch (\Throwable $e) {
            $this->log('Stripe status error', ['e'=>$e->getMessage()]);
        }
        return 'none';
    }

    /* ========== CRON ========== */

    public function schedule_cron() {
        if (!wp_next_scheduled('ch_normalize_company_states')) {
            wp_schedule_event(time()+120, 'twicedaily', 'ch_normalize_company_states');
        }
    }

    public function cron_normalize_all() {
        global $wpdb;
        // Empresas que necesitan revisión: en trial, o con sub/cust de stripe, o expiradas recientemente
        $rows = $wpdb->get_results("
            SELECT id FROM {$this->t_empresas}
            WHERE (trial_status='en_curso' AND trial_expires_at IS NOT NULL)
               OR (stripe_customer_id IS NOT NULL AND stripe_customer_id <> '')
               OR (stripe_subscription_id IS NOT NULL AND stripe_subscription_id <> '')
        ");
        if (!$rows) return;
        foreach ($rows as $r) {
            $this->normalize_empresa((int)$r->id);
        }
    }

    /* ========== WEBHOOK HANDLERS ========== */

    private function on_checkout_completed($session) {
        // sólo por coherencia: si desde aquí ya tienes customer/sub, normaliza
        $customer_id     = $session->customer     ?? null;
        $subscription_id = $session->subscription ?? null;

        if (!$customer_id) return;

        global $wpdb;
        // localizar empresa por customer_id o por CIF/email si lo tuvieras. Aquí: customer_id.
        $e = $wpdb->get_row($wpdb->prepare("
            SELECT id FROM {$this->t_empresas} WHERE stripe_customer_id=%s
        ", $customer_id));

        if ($e) {
            // guarda sub_id si viene
            if ($subscription_id) {
                $wpdb->update($this->t_empresas, ['stripe_subscription_id'=>$subscription_id], ['id'=>$e->id]);
            }
            $this->normalize_empresa((int)$e->id);
        }
    }

    private function on_subscription_upd($subscription) {
        $customer_id     = $subscription->customer ?? null;
        $subscription_id = $subscription->id ?? null;
        $status          = $subscription->status ?? null;
        if (!$customer_id) return;

        global $wpdb;
        $e = $wpdb->get_row($wpdb->prepare("
            SELECT id FROM {$this->t_empresas} WHERE stripe_customer_id=%s
        ", $customer_id));
        if (!$e) return;

        // guardamos sub_id por si cambia
        if ($subscription_id) {
            $wpdb->update($this->t_empresas, ['stripe_subscription_id'=>$subscription_id], ['id'=>$e->id]);
        }

        // normaliza en función del status
        if ($status === 'active') {
            $wpdb->update($this->t_empresas, [
                'trial_status'   => 'convertido',
                'estado_empresa' => 'activado',
            ], ['id'=>$e->id]);
        } elseif (in_array($status, ['canceled','unpaid','past_due','incomplete_expired'], true)) {
            // si el trial ya pasó, desactiva; si no, espera al fin del trial
            $row = $wpdb->get_row($wpdb->prepare("
                SELECT trial_status, trial_expires_at FROM {$this->t_empresas} WHERE id=%d
            ", $e->id));
            $now = current_time('timestamp');
            $exp = $row && $row->trial_expires_at ? strtotime(get_date_from_gmt($row->trial_expires_at)) : 0;

            if (!$row || $row->trial_status !== 'en_curso' || ($exp && $exp <= $now)) {
                $wpdb->update($this->t_empresas, [
                    'trial_status'   => 'expirado',
                    'estado_empresa' => 'desactivado',
                    'deactivated_at' => current_time('mysql'),
                ], ['id'=>$e->id]);
            }
        } else {
            // para otros estados, aseguramos normalización completa
            $this->normalize_empresa((int)$e->id);
        }
    }

    private function on_subscription_cancel($subscription) {
        $customer_id = $subscription->customer ?? null;
        if (!$customer_id) return;

        global $wpdb;
        $e = $wpdb->get_row($wpdb->prepare("SELECT id, trial_status, trial_expires_at FROM {$this->t_empresas} WHERE stripe_customer_id=%s", $customer_id));
        if (!$e) return;

        $now = current_time('timestamp');
        $exp = $e->trial_expires_at ? strtotime(get_date_from_gmt($e->trial_expires_at)) : 0;

        if ($e->trial_status !== 'en_curso' || ($exp && $exp <= $now)) {
            $wpdb->update($this->t_empresas, [
                'trial_status'   => 'expirado',
                'estado_empresa' => 'desactivado',
                'deactivated_at' => current_time('mysql'),
            ], ['id'=>$e->id]);
        }
    }

    private function on_invoice_failed($invoice) {
        $customer_id = $invoice->customer ?? null;
        if (!$customer_id) return;

        global $wpdb;
        $e = $wpdb->get_row($wpdb->prepare("SELECT id, trial_status, trial_expires_at FROM {$this->t_empresas} WHERE stripe_customer_id=%s", $customer_id));
        if (!$e) return;

        $now = current_time('timestamp');
        $exp = $e->trial_expires_at ? strtotime(get_date_from_gmt($e->trial_expires_at)) : 0;

        if ($e->trial_status !== 'en_curso' || ($exp && $exp <= $now)) {
            $wpdb->update($this->t_empresas, [
                'trial_status'   => 'expirado',
                'estado_empresa' => 'desactivado',
                'deactivated_at' => current_time('mysql'),
            ], ['id'=>$e->id]);
        }
    }
}

// Instancia única
new CH_Estado_Empresas();

/**
 * Función pública por si quieres normalizar una empresa concreta desde cualquier sitio:
 *  ch_normalize_empresa_estado(123);
 */
function ch_normalize_empresa_estado($empresa_id) {
    if (class_exists('CH_Estado_Empresas')) {
        $mgr = new CH_Estado_Empresas(); // crea contexto mínimo
        return $mgr->normalize_empresa((int)$empresa_id);
    }
    return false;
}

} // guard
