<?php
/**
 * Cron: recordatorios método de pago pendiente
 * Envía emails a los 15, 10, 5, 3 y 1 días antes de fin de trial si NO hay método de pago.
 * Usa plantilla: /templates-stripe/reminder-no-pm.html
 */
if (!defined('ABSPATH')) exit;

if (!class_exists('CH_Stripe_PM_Reminders')) {

class CH_Stripe_PM_Reminders {

    /** opción para marcar ejecuciones/sends y no duplicar */
    const OPT_PREFIX_SENT = 'ch_pm_reminder_'; // ch_pm_reminder_{empresa_id}_{days_left}

    /** opción caché PM (ya la usamos en admin) */
    const OPT_PREFIX_PM_CACHE = 'ch_pm_'; // ch_pm_{empresa_id} = ['has'=>0|1,'ts'=>date]

    /** días objetivo para enviar recordatorios */
    private $reminder_days = [15, 10, 5, 3, 1];

    public function __construct() {
        // Programar cron si no existe (diario)
        add_action('init', [$this, 'schedule_cron']);
        add_action('ch_stripe_check_pm_reminders', [$this, 'run']);
    }

    public function schedule_cron() {
        if (!wp_next_scheduled('ch_stripe_check_pm_reminders')) {
            // ejecuta en ~5 min y luego cada 12 horas (menos carga que hourly)
            wp_schedule_event(time() + 300, 'twicedaily', 'ch_stripe_check_pm_reminders');
        }
    }

    public function run() {
        global $wpdb;

        $t_empresas = 'mnkt_empresas';
        $t_rel      = 'mnkt_empresas_usuarios';
        $t_users    = 'mnkt_users';

        // ¿existen columnas para PM en la tabla?
        $has_pm_col = $this->table_has_column($t_empresas, 'has_payment_method');
        $now_ts     = current_time('timestamp'); // TZ WP

        // Selecciona empresas en trial en curso con fecha fin definida
        $sql = "
            SELECT e.id, e.user_cif, e.razon_social, e.trial_status, e.trial_expires_at,
                   e.stripe_customer_id, e.has_payment_method
            FROM {$t_empresas} e
            WHERE e.trial_status = 'en_curso'
              AND e.trial_expires_at IS NOT NULL
        ";
        $empresas = $wpdb->get_results($sql);

        if (!$empresas) return;

        foreach ($empresas as $e) {
            // calcular days_left desde trial_expires_at (guardado en UTC)
            $expires_local = $e->trial_expires_at ? strtotime(get_date_from_gmt($e->trial_expires_at)) : 0;
            if (!$expires_local) continue;

            $diff_days = (int) ceil( ($expires_local - $now_ts) / DAY_IN_SECONDS );

            // solo si encaja con nuestros días objetivo
            if (!in_array($diff_days, $this->reminder_days, true)) continue;

            // ¿ya enviado hoy para esta empresa y ese days_left?
            if ($this->already_sent($e->id, $diff_days)) continue;

            // ¿tiene método guardado?
            $has_pm = null;

            if ($has_pm_col && isset($e->has_payment_method)) {
                $has_pm = (int)$e->has_payment_method === 1;
            } else {
                $cache = get_option(self::OPT_PREFIX_PM_CACHE . $e->id);
                if (is_array($cache) && array_key_exists('has', $cache)) {
                    $has_pm = (int)$cache['has'] === 1;
                }
            }

            // Si no sabemos, intentamos comprobar en Stripe (suave)
            if ($has_pm === null) {
                $has_pm = $this->check_pm_in_stripe_and_cache($e->id, $e->stripe_customer_id, $has_pm_col);
            }

            // si ya tiene método, no molestamos
            if ($has_pm === true) continue;

            // obtener email y nombre del gestor principal
            $gestor = $wpdb->get_row($wpdb->prepare("
                SELECT u.user_email, COALESCE(NULLIF(u.display_name,''), u.user_login) AS name, u.user_login
                FROM {$t_rel} r
                INNER JOIN {$t_users} u ON u.ID = r.user_id
                WHERE r.empresa_id = %d AND r.rol_en_empresa = 'Gestor' AND r.estado='Activado'
                ORDER BY r.creado_en ASC
                LIMIT 1
            ", $e->id));

            if (!$gestor || empty($gestor->user_email)) continue;

            // construir link al portal
            $portal_link = add_query_arg('ch_stripe_portal', '1', site_url('/'));

            // preparar variables de la plantilla
            $vars = [
                'name'        => $gestor->name ?: $gestor->user_login,
                'email'       => $gestor->user_email,
                'empresa'     => $e->razon_social ?: '',
                'cif'         => $e->user_cif,
                'days_left'   => $diff_days,
                'trial_end'   => date_i18n('d/m/Y H:i', $expires_local),
                'portal_link' => $portal_link,
            ];

            // enviar
            $sent = $this->send_reminder_email($vars);
            if ($sent) {
                $this->mark_sent($e->id, $diff_days);
            }
        }
    }

    /* ========================= Helpers ========================= */

    private $table_columns_cache = [];

    private function table_has_column($table, $column) {
        global $wpdb;
        $k = $table.'::'.$column;
        if (isset($this->table_columns_cache[$k])) return $this->table_columns_cache[$k];
        $exists = (bool) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = %s AND COLUMN_NAME = %s",
            $table, $column
        ));
        $this->table_columns_cache[$k] = $exists;
        return $exists;
    }

    private function check_pm_in_stripe_and_cache($empresa_id, $customer_id, $has_pm_col) {
        if (!$customer_id) return false;

        // Carga SDK
        $base = defined('CH_STRIPE_WH_PLUGIN_DIR') ? CH_STRIPE_WH_PLUGIN_DIR : trailingslashit(dirname(__DIR__));
        $sdk  = $base . 'lib/stripe-php/init.php';
        if (file_exists($sdk)) require_once $sdk;
        if (!class_exists('\Stripe\StripeClient')) return false;

        // API secret desde constante o opción
        $secret = defined('CONTROL_HORARIO_STRIPE_SECRET_KEY') ? CONTROL_HORARIO_STRIPE_SECRET_KEY : get_option('ch_stripe_secret_key', '');
        if (!$secret) return false;

        $has_pm = false;
        try {
            $client = new \Stripe\StripeClient($secret);

            // a) default pm
            $cus = $client->customers->retrieve($customer_id, []);
            if ($cus && !empty($cus->invoice_settings) && !empty($cus->invoice_settings->default_payment_method)) {
                $has_pm = true;
            }

            // b) lista PMs si sigue false
            if (!$has_pm) {
                $methods = $client->paymentMethods->all([
                    'customer' => $customer_id,
                    'type'     => 'card',
                    'limit'    => 1,
                ]);
                $has_pm = !empty($methods->data);
            }
        } catch (\Throwable $e) {
            // mejor no romper; asume false si hay error
            $has_pm = false;
        }

        // persistir
        global $wpdb;
        $t = 'mnkt_empresas';

        if ($has_pm_col) {
            $wpdb->update($t, [
                'has_payment_method'          => $has_pm ? 1 : 0,
                'payment_method_last_checked' => current_time('mysql'),
            ], ['id' => $empresa_id]);
        } else {
            update_option(self::OPT_PREFIX_PM_CACHE . $empresa_id, [
                'has' => $has_pm ? 1 : 0,
                'ts'  => current_time('mysql'),
            ], false);
        }

        return $has_pm;
    }

    private function already_sent($empresa_id, $days_left) {
        $key = self::OPT_PREFIX_SENT . intval($empresa_id) . '_' . intval($days_left);
        $val = get_option($key);
        // si existe y es del mismo día, no re-enviar
        if (!$val) return false;
        $last = strtotime($val);
        // comparar fecha (no hora)
        return (date_i18n('Y-m-d', $last) === date_i18n('Y-m-d', current_time('timestamp')));
    }

    private function mark_sent($empresa_id, $days_left) {
        $key = self::OPT_PREFIX_SENT . intval($empresa_id) . '_' . intval($days_left);
        update_option($key, current_time('mysql'), false);
    }

    private function send_reminder_email($vars) {
        // cargar plantilla
        $base = defined('CH_STRIPE_WH_PLUGIN_DIR') ? CH_STRIPE_WH_PLUGIN_DIR : trailingslashit(dirname(__DIR__));
        $tpl  = trailingslashit($base . 'templates-stripe') . 'reminder-no-pm.html';

        if (file_exists($tpl)) {
            $html = file_get_contents($tpl);
        } else {
            // fallback minimal
            $html = '
                <h2>Recordatorio de método de pago</h2>
                <p>Hola {{name}}, todavía no tenemos un método de pago guardado para {{empresa}} ({{cif}}).</p>
                <p>Quedan <strong>{{days_left}}</strong> día(s). El periodo de prueba termina el <strong>{{trial_end}}</strong>.</p>
                <p><a href="{{portal_link}}">Añadir/actualizar método de pago</a></p>
            ';
        }

        // reemplazos
        $repl = [
            '{{name}}'        => esc_html($vars['name'] ?? ''),
            '{{empresa}}'     => esc_html($vars['empresa'] ?? ''),
            '{{cif}}'         => esc_html($vars['cif'] ?? ''),
            '{{days_left}}'   => esc_html($vars['days_left'] ?? ''),
            '{{trial_end}}'   => esc_html($vars['trial_end'] ?? ''),
            '{{portal_link}}' => esc_url($vars['portal_link'] ?? site_url('/')),
        ];
        $body = strtr($html, $repl);

        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $subject = 'Recordatorio: añade tu método de pago';

        return wp_mail($vars['email'], $subject, $body, $headers);
    }
}

// Instanciar
new CH_Stripe_PM_Reminders();

} // class guard
