<?php
if (!defined('ABSPATH')) exit;

/**
 * Webhook Stripe (procedural) para:
 *  - GET  /wp-json/control-horario/v1/stripe-webhook (ping)
 *  - POST /wp-json/control-horario/v1/stripe-webhook (Stripe real)
 *
 * Requiere:
 *  - define('CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET', 'whsec_xxx') en wp-config.php
 *  - SDK stripe-php en:
 *      /lib/stripe-php/init.php   (recomendado)
 *    o /vendor/autoload.php       (si usas Composer)
 *
 * Tablas:
 *  - mnkt_empresas (user_cif varchar(20), enums reales que pasaste)
 *  - mnkt_users (user_cif varchar(10), user_dni varchar(10))
 *  - mnkt_empresas_usuarios (UNIQUE (user_id, empresa_id, rol_en_empresa))
 */

/* ==================== UTILIDADES ==================== */

function ch_sw_log($msg, $ctx = []) {
    $line = '['.date('Y-m-d H:i:s').'] '.$msg.(empty($ctx)?'':' | '.json_encode($ctx, JSON_UNESCAPED_UNICODE));
    if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
        error_log('CH_STRIPE_WEBHOOK: '.$line);
    }
    $up = wp_get_upload_dir();
    @file_put_contents(trailingslashit($up['basedir']).'control-horario-webhook.log', $line.PHP_EOL, FILE_APPEND);
}

function ch_sw_try_require($path, $label){
    if (file_exists($path)) {
        require_once $path;
        ch_sw_log('SDK intento', ['label'=>$label, 'path'=>$path, 'ok'=>class_exists('\Stripe\Webhook')]);
        return class_exists('\Stripe\Webhook');
    }
    ch_sw_log('SDK no existe', ['label'=>$label, 'path'=>$path]);
    return false;
}

function ch_sw_load_sdk(){
    if (class_exists('\Stripe\Webhook')) return true;

    $candidates = [];

    // a) usando constante del plugin principal si existe
    if (defined('CH_STRIPE_WH_PLUGIN_DIR')) {
        $candidates[] = [CH_STRIPE_WH_PLUGIN_DIR . 'lib/stripe-php/init.php', 'CH_STRIPE_WH_PLUGIN_DIR/lib'];
        $candidates[] = [CH_STRIPE_WH_PLUGIN_DIR . 'vendor/autoload.php', 'CH_STRIPE_WH_PLUGIN_DIR/vendor'];
    }

    // b) relativas a este archivo (/includes/)
    $base_inc = plugin_dir_path(__FILE__);
    $candidates[] = [dirname($base_inc) . '/lib/stripe-php/init.php', 'plugin_root/lib'];
    $candidates[] = [dirname($base_inc) . '/vendor/autoload.php', 'plugin_root/vendor'];

    // c) dos niveles por si acaso
    $candidates[] = [dirname(dirname($base_inc)) . '/lib/stripe-php/init.php', 'upper/lib'];
    $candidates[] = [dirname(dirname($base_inc)) . '/vendor/autoload.php', 'upper/vendor'];

    foreach ($candidates as [$p,$label]) {
        if (ch_sw_try_require($p, $label)) return true;
    }
    return class_exists('\Stripe\Webhook');
}

function ch_sw_db_error($ctx=''){
    global $wpdb;
    if (!empty($wpdb->last_error)) {
        ch_sw_log('DB ERROR '.$ctx, ['err'=>$wpdb->last_error, 'last_query'=>$wpdb->last_query]);
    }
}

function ch_sw_only_digits($s){ return preg_replace('/\D+/', '', (string)$s); }

function ch_sw_send_welcome_email($vars){
    // $vars: email, name, empresa, cif, user_login, temp_password (opcional), trial_end (opcional)

    // Genera link de restablecer contraseña
    $reset_link = '';
    $user = get_user_by('login', $vars['user_login'] ?? '');
    if ($user && function_exists('get_password_reset_key')) {
        $key = get_password_reset_key($user);
        if (!is_wp_error($key)) {
            $reset_link = wp_login_url() . "?action=rp&key=" . rawurlencode($key) . "&login=" . rawurlencode($user->user_login);
        }
    }
    if (!$reset_link) { $reset_link = wp_lostpassword_url(); }

    $dashboard_url = site_url('/');

    // Localiza la plantilla welcome.html
    $base = defined('CH_STRIPE_WH_PLUGIN_DIR') ? CH_STRIPE_WH_PLUGIN_DIR : plugin_dir_path(dirname(__FILE__));
    $template_path = trailingslashit($base) . 'templates-stripe/welcome.html';

    if (file_exists($template_path)) {
        $html = file_get_contents($template_path);
    } else {
        // Fallback simple si no encuentra plantilla
        $html = '
            <h2>¡Bienvenido/a a Control Horario!</h2>
            <p>Hola {{name}}, hemos creado tu acceso como Gestor para {{empresa}} ({{cif}}).</p>
            <p><strong>Usuario:</strong> {{user_login}}<br>
            {{temp_password_block}}</p>
            <p><a href="{{reset_link}}">Establecer contraseña</a></p>
            <p>Tu periodo de prueba termina el: <strong>{{trial_end}}</strong></p>
            <p><a href="{{dashboard_url}}">Ir al panel</a></p>';
    }

    // Bloque de contraseña temporal (solo si existe)
    $temp_password_block = '';
    if (!empty($vars['temp_password'])) {
        $temp_password_block = '<strong>Contraseña temporal:</strong> ' . esc_html($vars['temp_password']);
    }

    // Reemplazos
    $replacements = [
        '{{name}}'          => esc_html( wp_strip_all_tags($vars['name'] ?? '') ),
        '{{empresa}}'       => esc_html( wp_strip_all_tags($vars['empresa'] ?? '') ),
        '{{cif}}'           => esc_html( wp_strip_all_tags($vars['cif'] ?? '') ),
        '{{user_login}}'    => esc_html( wp_strip_all_tags($vars['user_login'] ?? '') ),
        '{{temp_password}}' => esc_html( wp_strip_all_tags($vars['temp_password'] ?? '') ),
        '{{temp_password_block}}' => $temp_password_block,
        '{{reset_link}}'    => esc_url($reset_link),
        '{{trial_end}}'     => esc_html( wp_strip_all_tags($vars['trial_end'] ?? '') ),
        '{{dashboard_url}}' => esc_url($dashboard_url),
    ];
    $body = strtr($html, $replacements);

    $subject = 'Bienvenido/a a Control Horario';
    $headers = ['Content-Type: text/html; charset=UTF-8'];
    // (Opcional) pon un remitente bonito:
    // $headers[] = 'From: Control Horario <no-reply@educasede.com>';

    ch_sw_log('EMAIL: intentando enviar welcome', ['to'=>$vars['email'], 'template_exists'=>file_exists($template_path)]);

    wp_mail($vars['email'], $subject, $body, $headers);
}


/* ==================== CARGA SDK TEMPRANA ==================== */

add_action('plugins_loaded', function () { ch_sw_load_sdk(); });

/* ==================== ENDPOINT REST ==================== */

add_action('rest_api_init', function () {
    register_rest_route('control-horario/v1', '/stripe-webhook', [
        'methods'  => ['POST','GET'],
        'callback' => 'ch_sw_webhook_handler',
        'permission_callback' => '__return_true',
    ]);
});

/* ==================== ROUTER ==================== */

function ch_sw_webhook_handler(\WP_REST_Request $req){
    // GET -> ping
    if ($req->get_method()==='GET') {
        ch_sw_log('PING stripe-webhook ok', ['uri'=>$_SERVER['REQUEST_URI'] ?? '']);
        return new \WP_REST_Response(['ok'=>true,'ping'=>'stripe-webhook'],200);
    }

    // POST -> Stripe
    // reintento de carga del SDK por si plugins_loaded no lo pilló
    if (!class_exists('\Stripe\Webhook')) ch_sw_load_sdk();

    if (!class_exists('\Stripe\Webhook')) {
        ch_sw_log('Stripe SDK no cargado (tras reintento)');
        return new \WP_REST_Response(['ok'=>false,'error'=>'Stripe SDK no cargado'],500);
    }
    if (!defined('CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET') || !CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET) {
        ch_sw_log('Secret no definido');
        return new \WP_REST_Response(['ok'=>false,'error'=>'CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET no definido'],500);
    }

    $payload = $req->get_body();
    $sig     = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
    ch_sw_log('POST recibido', ['sig'=>(bool)$sig, 'len'=>strlen($payload)]);

    try {
        $event = \Stripe\Webhook::constructEvent($payload, $sig, CONTROL_HORARIO_STRIPE_WEBHOOK_SECRET);
    } catch (\UnexpectedValueException $e) {
        ch_sw_log('Payload inválido', ['e'=>$e->getMessage()]);
        return new \WP_REST_Response(['ok'=>false,'error'=>'Payload inválido'],400);
    } catch (\Stripe\Exception\SignatureVerificationException $e) {
        ch_sw_log('Firma inválida', ['e'=>$e->getMessage()]);
        return new \WP_REST_Response(['ok'=>false,'error'=>'Firma inválida'],400);
    } catch (\Throwable $e) {
        ch_sw_log('constructEvent ERROR', ['e'=>$e->getMessage()]);
        return new \WP_REST_Response(['ok'=>false,'error'=>$e->getMessage()],400);
    }

    ch_sw_log('Evento OK', ['type'=>$event->type]);

    if ($event->type === 'checkout.session.completed') {
        $out = ch_sw_process_session($event->data->object);
        ch_sw_log('Process result', $out);
        // devolvemos 200 siempre para Stripe
        return new \WP_REST_Response(['ok'=>true], 200);
    }

    return new \WP_REST_Response(['ok'=>true], 200);
}

/* ==================== LÓGICA: EMPRESA + USER + RELACIÓN ==================== */

function ch_sw_process_session($session){
    global $wpdb;

    // Log snapshot
    try {
        ch_sw_log('SESSION SNAPSHOT', [
            'id'             => $session->id ?? null,
            'payment_status' => $session->payment_status ?? null,
            'mode'           => $session->mode ?? null,
            'payment_link'   => $session->payment_link ?? null,
            'subscription'   => $session->subscription ?? null,
            'customer'       => $session->customer ?? null,
            'email'          => $session->customer_details->email ?? ($session->customer_email ?? null),
            'business_name'  => $session->customer_details->business_name ?? ($session->collected_information->business_name ?? null),
        ]);
    } catch (\Throwable $e) {
        ch_sw_log('SNAPSHOT ERROR', ['e'=>$e->getMessage()]);
    }

    // 1) Extraer + normalizar
    $email = $session->customer_details->email ?? ($session->customer_email ?? '');
    $email = sanitize_email($email);
    $customer_id     = $session->customer     ?? null;
    $subscription_id = $session->subscription ?? null;

    // custom_fields -> kv normalizado
    $cf = [];
    foreach ((array)($session->custom_fields ?? []) as $f) {
        $key = '';
        if (!empty($f->key)) $key = $f->key;
        if (!$key && !empty($f->label->custom)) $key = $f->label->custom;
        $key = strtolower(preg_replace('/[^a-z0-9]+/','', (string)$key));
        $val = $f->text->value ?? '';
        if ($key && $val!=='') $cf[$key] = (string)$val;
    }
    // helper para leer desde cf por alias / contiene
    $pick = function($aliasesExactos, $contiene=[], $excluir=[]) use($cf){
        foreach ($aliasesExactos as $a) {
            if (isset($cf[$a]) && $cf[$a] !== '') return trim($cf[$a]);
        }
        foreach ($cf as $k=>$v){
            if ($v==='') continue;
            $skip=false;
            foreach ($excluir as $bad){ if ($bad && strpos($k,$bad)!==false){ $skip=true; break; } }
            if ($skip) continue;
            foreach ($contiene as $frag){ if ($frag && strpos($k,$frag)!==false) return trim($v); }
        }
        return '';
    };

    $cif_in   = $pick(['cif','cifempresa','cifempresaodniautonomo','cifempresaodniautnomo','dni','dniautonomo']);
    $nombre_g = $pick(['nombregestor','nombregestora','nombredelgestor']);
    $dni_in   = ch_sw_only_digits($pick(['dnigestor','dnigestorasinletra','dni']));

    $razon_social = $pick(['razonsocial','razon','razon_social','nombrefiscal','razonsocialfiscal'], ['razon','social','fiscal'], ['cif','dni']);
    if (!$razon_social) { $razon_social = $session->customer_details->business_name ?? ($session->collected_information->business_name ?? ''); }
    if (!$razon_social) { $razon_social = $session->customer_details->name ?? ''; }

    if (!$email) {
        ch_sw_log('ABORT: sin email');
        return ['ok'=>false,'error'=>'sin email'];
    }
    if (!$cif_in) {
        $cif_in = 'PEND-' . strtoupper(substr(md5($email . time()), 0, 8));
        ch_sw_log('CIF provisional', ['cif'=>$cif_in]);
    }

    // BD: empresas.user_cif (20), users.user_cif/user_dni (10)
    $cif_raw = strtoupper(trim($cif_in));
    $cif20   = (strlen($cif_raw)>20) ? substr($cif_raw,0,20) : $cif_raw; // empresas
    $cif10   = substr($cif20, 0, 10);                                    // users
    $dni10   = $dni_in ? substr($dni_in, 0, 10) : $dni_in;

    ch_sw_log('PRE-UP', [
        'email'=>$email,'cif_raw'=>$cif_raw,'cif20'=>$cif20,'cif10'=>$cif10,'dni10'=>$dni10,
        'razon'=>$razon_social,'customer_id'=>$customer_id,'sub_id'=>$subscription_id
    ]);

    // 2) EMPRESA
    $t_empresas = 'mnkt_empresas';
    $t_users    = 'mnkt_users';
    $t_rel      = 'mnkt_empresas_usuarios';

    $empresa_id = (int)$wpdb->get_var($wpdb->prepare("SELECT id FROM {$t_empresas} WHERE user_cif=%s", $cif20));

    $empresa_data = [
        'user_cif'                    => $cif20,
        'razon_social'                => $razon_social,
        'estado_empresa'              => 'activado',
        'estado_validacion'           => 'Pendiente',
        'fecha_aceptacion'            => current_time('mysql'),
        'ip_firma'                    => $_SERVER['REMOTE_ADDR'] ?? '',
        'user_agent'                  => substr($_SERVER['HTTP_USER_AGENT'] ?? 'Stripe', 0, 255),
        'gamificacion'                => 'activado',
        'aceptacion_proteccion_datos' => 1,
        'metodo_pago'                 => 'Tarjeta',
        'stripe_customer_id'          => $customer_id,
        'stripe_subscription_id'      => $subscription_id,
        'trial_status'                => 'en_curso',
        'trial_started_at'            => current_time('mysql'),
        'trial_days'                  => 35,
        'trial_expires_at'            => gmdate('Y-m-d H:i:s', strtotime(current_time('mysql')) + (35 * DAY_IN_SECONDS)),
    ];

    if ($empresa_id) {
        if (!$empresa_data['razon_social']) unset($empresa_data['razon_social']);
        $wpdb->update($t_empresas, $empresa_data, ['id'=>$empresa_id]);
        ch_sw_db_error('update empresa');
        ch_sw_log('EMPRESA UPDATE', ['empresa_id'=>$empresa_id]);
    } else {
        $wpdb->insert($t_empresas, $empresa_data);
        ch_sw_db_error('insert empresa');
        $empresa_id = (int)$wpdb->insert_id;
        ch_sw_log('EMPRESA INSERT', ['empresa_id'=>$empresa_id]);
        if (!$empresa_id) return ['ok'=>false,'error'=>'insert empresa sin id'];
    }

    // 3) USUARIO (Gestor)
    $user_id = (int)$wpdb->get_var($wpdb->prepare("SELECT ID FROM {$t_users} WHERE user_email=%s", $email));
    $just_created = false;
    $user_login   = '';
    $temp_password= null;

    if ($user_id) {
        $user_login = (string)$wpdb->get_var($wpdb->prepare("SELECT user_login FROM {$t_users} WHERE ID=%d", $user_id));
        $display    = $nombre_g ?: (string)$wpdb->get_var($wpdb->prepare("SELECT display_name FROM {$t_users} WHERE ID=%d", $user_id));

        $wpdb->update($t_users, [
            'display_name' => $display,
            'user_dni'     => $dni10,
            'user_cif'     => $cif10,
            'user_cif_id'  => $empresa_id,
            'user_rol'     => 'Gestor',
            'user_estado'  => 'Activado',
            'user_gamificacion'=> 'Activado',
        ], ['ID'=>$user_id]);
        ch_sw_db_error('update user');
        ch_sw_log('USER UPDATE', ['user_id'=>$user_id,'login'=>$user_login]);
    } else {
       // usar el mismo nombre que display_name para el user_login
            $login_base = $nombre_g ?: current(explode('@', $email));

            // quita acentos y deja formato "seguro" para login
            $login_base = remove_accents($login_base);
            $user_login = sanitize_user($login_base, true);

            // fallback si tras sanear se queda vacío
            if (!$user_login) { $user_login = 'gestor'; }

            // límite WP: 60 chars
            $user_login = substr($user_login, 0, 60);

            // garantizar unicidad
            $exists = $wpdb->get_var($wpdb->prepare("SELECT ID FROM {$t_users} WHERE user_login=%s", $user_login));
            if ($exists) {
                for ($i = 2; $i < 1000; $i++) {
                    $try = substr($user_login, 0, 60 - strlen((string)$i)) . $i;
                    $exists = $wpdb->get_var($wpdb->prepare("SELECT ID FROM {$t_users} WHERE user_login=%s", $try));
                    if (!$exists) { $user_login = $try; break; }
                }
            }

        $temp_password = wp_generate_password(10, false);
        $hash          = wp_hash_password($temp_password);

        $wpdb->insert($t_users, [
            'user_login'      => $user_login,
            'user_pass'       => $hash,
            'user_nicename'   => $nombre_g ?: $user_login,
            'user_email'      => $email,
            'user_registered' => current_time('mysql'),
            'display_name'    => $nombre_g ?: $user_login,
            'user_dni'        => $dni10,
            'user_cif'        => $cif10,
            'user_cif_id'     => $empresa_id,
            'user_rol'        => 'Gestor',
            'user_estado'     => 'Activado',
            'user_gamificacion'=> 'Activado', 
        ]);
        ch_sw_db_error('insert user');
        $user_id = (int)$wpdb->insert_id;
        ch_sw_log('USER INSERT', ['user_id'=>$user_id,'login'=>$user_login]);
        if (!$user_id) return ['ok'=>false,'error'=>'insert user sin id'];
        $just_created = true;

        // Email bienvenida si es nuevo
if ($just_created) {
    // calcula fin de trial (opcional, si quieres mostrarlo en el correo)
    $trial_expires_at = $wpdb->get_var($wpdb->prepare("SELECT trial_expires_at FROM mnkt_empresas WHERE id=%d", $empresa_id));
    $trial_end = '';
    if ($trial_expires_at) {
        $timestamp = strtotime(get_date_from_gmt($trial_expires_at));
        $trial_end = date_i18n('d/m/Y H:i', $timestamp);
    }

    try {
        ch_sw_send_welcome_email([
            'email'         => $email,
            'name'          => $nombre_g ?: $user_login,
            'empresa'       => $razon_social ?: '',
            'cif'           => $cif10,           // para mostrar, usamos el corto (tu users usa 10). Si prefieres el largo, usa $cif20
            'user_login'    => $user_login,
            'temp_password' => $temp_password,   // solo si es nuevo
            'trial_end'     => $trial_end,
        ]);
        ch_sw_log('EMAIL: welcome enviado', ['to'=>$email]);
    } catch (\Throwable $e) {
        ch_sw_log('EMAIL: welcome ERROR', ['e'=>$e->getMessage(), 'to'=>$email]);
    }
}

    }

    // 4) RELACIÓN
    $exists_rel = (int)$wpdb->get_var($wpdb->prepare(
        "SELECT id FROM {$t_rel} WHERE user_id=%d AND empresa_id=%d AND rol_en_empresa='Gestor'",
        $user_id, $empresa_id
    ));
    if (!$exists_rel) {
        $wpdb->insert($t_rel, [
            'user_id'        => $user_id,
            'empresa_id'     => $empresa_id,
            'rol_en_empresa' => 'Gestor',
            'estado'         => 'Activado',
            'creado_en'      => current_time('mysql'),
        ]);
        ch_sw_db_error('insert rel');
        ch_sw_log('REL INSERT', ['user_id'=>$user_id,'empresa_id'=>$empresa_id]);
    } else {
        ch_sw_log('REL EXISTS', ['rel_id'=>$exists_rel]);
    }

    ch_sw_log('DONE', ['empresa_id'=>$empresa_id,'user_id'=>$user_id]);

    return [
        'ok'           => true,
        'empresa_id'   => $empresa_id,
        'user_id'      => $user_id,
        'email'        => $email,
        'cif20'        => $cif20,
        'cif10'        => $cif10,
        'created_user' => $just_created,1
    ];
}
