<?php
if (!defined('ABSPATH')) exit;

add_shortcode('listar_trabajadores_m2m', 'mostrar_trabajadores_m2m');

function mostrar_trabajadores_m2m() {
    if (!is_user_logged_in()) {
        return '<p class="pure-alert" style="text-align: center;">No has iniciado sesión.</p>';
    }

    global $wpdb;

    // Datos del usuario actual (en tu tabla mnkt_users)
    $wp_user = wp_get_current_user();
    $usuario_actual = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM mnkt_users WHERE user_email = %s",
        $wp_user->user_email
    ));

    if (!$usuario_actual) {
        return '<p>No se encontró un usuario válido.</p>';
    }

    // Solo Gestor Activado
    if ($usuario_actual->user_rol !== 'Gestor' || $usuario_actual->user_estado !== 'Activado') {
        return '<p class="pure-alert" style="text-align: center;">No tienes permisos para acceder a esta funcionalidad.</p>';
    }

    // ===== NUEVO: empresas del gestor vía M2M =====
    // Tablas
    $t_users = 'mnkt_users';
    $t_emp   = 'mnkt_empresas';
    $t_m2m   = 'mnkt_empresas_usuarios';

    // Empresas asociadas al gestor (por M2M)
    $empresas = $wpdb->get_results($wpdb->prepare("
        SELECT e.id, e.razon_social, e.user_cif
        FROM $t_emp e
        JOIN $t_m2m eu ON eu.empresa_id = e.id
        WHERE eu.user_id = %d
        ORDER BY e.razon_social ASC
    ", $usuario_actual->ID));

    if (empty($empresas)) {
        return '<p class="pure-alert">No tienes ninguna empresa asociada. Pide a un admin que te vincule en el panel.</p>';
    }

    // Determinar empresa activa: por GET empresa_id (si pertenece al gestor), o la primera
    $empresa_ids = array_map(fn($e) => intval($e->id), $empresas);
    $empresa_sel_id = isset($_GET['empresa_id']) ? intval($_GET['empresa_id']) : 0;
    if (!$empresa_sel_id || !in_array($empresa_sel_id, $empresa_ids, true)) {
        $empresa_sel_id = $empresas[0]->id;
    }

    // Obtener datos de la empresa seleccionada
    $empresa_sel = null;
    foreach ($empresas as $e) {
        if (intval($e->id) === intval($empresa_sel_id)) { $empresa_sel = $e; break; }
    }
    if (!$empresa_sel) {
        return '<p class="pure-alert">No se pudo determinar la empresa activa.</p>';
    }

    // Filtro de estado (como en tu código original)
    $estado_filtro = isset($_GET['estado']) ? sanitize_text_field($_GET['estado']) : 'todos';
    $condicion_estado = '';
    if ($estado_filtro === 'Activado') {
        $condicion_estado = "AND user_estado = 'Activado'";
    } elseif ($estado_filtro === 'Desactivado') {
        $condicion_estado = "AND user_estado = 'Desactivado'";
    }

    // ===== Acciones (toggle estado / reenviar / editar) – igual que lo tenías =====
    // Toggle estado trabajador
    if (isset($_GET['toggle'])) {
        $id = intval($_GET['toggle']);
        // seguridad mínima: que el trabajador pertenezca a la empresa seleccionada
        $cif_trab = $wpdb->get_var($wpdb->prepare("SELECT user_cif FROM mnkt_users WHERE ID = %d", $id));
        if ($cif_trab && $cif_trab === $empresa_sel->user_cif) {
            $estado_actual = $wpdb->get_var($wpdb->prepare("SELECT user_estado FROM mnkt_users WHERE ID = %d", $id));
            $nuevo_estado = ($estado_actual === 'Activado') ? 'Desactivado' : 'Activado';
            $wpdb->update('mnkt_users', ['user_estado' => $nuevo_estado], ['ID' => $id]);
        }
        // limpiar param y mantener filtros
        $qs = $_GET; unset($qs['toggle']);
        wp_safe_redirect(add_query_arg($qs, remove_query_arg('toggle')));
        exit;
    }

    // Reenviar recuperación
    if (isset($_GET['reenviar']) && is_numeric($_GET['reenviar'])) {
        $trabajador = get_userdata(intval($_GET['reenviar']));
        if ($trabajador && $trabajador->user_email) {
            retrieve_password($trabajador->user_login);
            echo '<p class="pure-alert pure-alert-success" style="margin-top: 10px;">🔑 Enlace de recuperación enviado a ' . esc_html($trabajador->user_email) . '</p>';
        }
    }

    // Editar trabajador (como lo tenías)
    if (isset($_POST['editar_trabajador'])) {
        $id = intval($_POST['trabajador_id']);
        $nombre = sanitize_text_field($_POST['nombre']);
        $email = sanitize_email($_POST['email']);
        $dni = sanitize_text_field($_POST['dni']);

        // validar que el trabajador es de la empresa seleccionada
        $cif_trab = $wpdb->get_var($wpdb->prepare("SELECT user_cif FROM mnkt_users WHERE ID = %d", $id));
        if ($cif_trab !== $empresa_sel->user_cif) {
            return '<p style="color:red;">Error: No puedes editar trabajadores de otra empresa.</p>';
        }

        $existe_email = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM mnkt_users WHERE user_email = %s AND ID != %d", $email, $id));
        if ($existe_email > 0) {
            echo '<p style="color:red;">Error: El email ya está en uso.</p>';
        } else {
            $wpdb->update('mnkt_users', [
                'display_name' => $nombre,
                'user_email' => $email,
                'user_dni' => $dni
            ], ['ID' => $id]);
            // recargar manteniendo filtros
            wp_safe_redirect($_SERVER['REQUEST_URI']);
            exit;
        }
    }

    // ===== Listado de trabajadores de la empresa seleccionada =====
    $trabajadores = $wpdb->get_results($wpdb->prepare("
        SELECT * FROM mnkt_users
        WHERE user_cif = %s
          AND user_rol = 'Trabajador'
          $condicion_estado
        ORDER BY display_name ASC
    ", $empresa_sel->user_cif));

    // ===== Render =====
    ob_start();
    ?>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/pure/2.0.6/pure-min.css">

    <h3>Listar Trabajadores (M2M)</h3>

    <?php if (count($empresas) > 1): ?>
        <form method="get" class="pure-form" style="display:flex; gap:12px; align-items:center; flex-wrap:wrap; margin-bottom: 15px;">
            <?php
            // preservar otros parámetros útiles
            foreach ($_GET as $k=>$v) {
                if (in_array($k, ['empresa_id','estado'], true)) continue;
                echo '<input type="hidden" name="'.esc_attr($k).'" value="'.esc_attr($v).'">';
            }
            ?>
            <label>Empresa:
                <select name="empresa_id" onchange="this.form.submit()">
                    <?php foreach ($empresas as $e): ?>
                        <option value="<?php echo esc_attr($e->id); ?>" <?php selected($e->id, $empresa_sel_id); ?>>
                            <?php echo esc_html($e->razon_social ?: ('Empresa #'.$e->id)); ?>
                            <?php echo $e->user_cif ? ' — '.esc_html($e->user_cif) : ''; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </label>

            <label>Estado:
                <select name="estado" onchange="this.form.submit()">
                    <option value="todos" <?php selected($estado_filtro, 'todos'); ?>>Todos</option>
                    <option value="Activado" <?php selected($estado_filtro, 'Activado'); ?>>Activados</option>
                    <option value="Desactivado" <?php selected($estado_filtro, 'Desactivado'); ?>>Desactivados</option>
                </select>
            </label>
        </form>
    <?php else: ?>
        <p><strong>Empresa:</strong>
            <?php echo esc_html($empresa_sel->razon_social ?: ('Empresa #'.$empresa_sel->id)); ?>
            <?php echo $empresa_sel->user_cif ? ' — CIF: '.esc_html($empresa_sel->user_cif) : ''; ?>
        </p>
        <form method="get" class="pure-form" style="margin-bottom: 15px;">
            <?php
            foreach ($_GET as $k=>$v) {
                if ($k === 'estado') continue;
                echo '<input type="hidden" name="'.esc_attr($k).'" value="'.esc_attr($v).'">';
            }
            ?>
            <label>Estado:
                <select name="estado" onchange="this.form.submit()">
                    <option value="todos" <?php selected($estado_filtro, 'todos'); ?>>Todos</option>
                    <option value="Activado" <?php selected($estado_filtro, 'Activado'); ?>>Activados</option>
                    <option value="Desactivado" <?php selected($estado_filtro, 'Desactivado'); ?>>Desactivados</option>
                </select>
            </label>
        </form>
    <?php endif; ?>

    <table class="pure-table pure-table-bordered">
        <thead>
            <tr>
                <th>Nombre</th>
                <th>Email</th>
                <th>DNI</th>
                <?php if ($estado_filtro === 'todos') : ?>
                    <th>Estado</th>
                <?php endif; ?>
                <th>Acciones</th>
            </tr>
        </thead>
        <tbody>
        <?php if (empty($trabajadores)): ?>
            <tr><td colspan="<?php echo ($estado_filtro === 'todos') ? 5 : 4; ?>">No hay trabajadores con ese filtro.</td></tr>
        <?php else: ?>
            <?php foreach ($trabajadores as $trabajador) : ?>
                <tr>
                    <td><?php echo esc_html($trabajador->display_name); ?></td>
                    <td><?php echo esc_html($trabajador->user_email); ?></td>
                    <td><?php echo esc_html($trabajador->user_dni); ?></td>

                    <?php if ($estado_filtro === 'todos') : ?>
                        <td>
                            <?php
                                $estado = esc_html($trabajador->user_estado);
                                $color = ($estado === 'Activado') ? 'green' : 'red';
                                echo "<span style='color: $color; font-weight: bold;'>$estado</span>";
                            ?>
                        </td>
                    <?php endif; ?>

                    <td>
                        <?php
                        // construir enlaces preservando empresa/estado
                        $base = add_query_arg([
                            'empresa_id' => $empresa_sel_id,
                            'estado'     => $estado_filtro,
                        ], remove_query_arg(['toggle','edit','reenviar']));
                        ?>
                        <a href="<?php echo esc_url(add_query_arg('edit', $trabajador->ID, $base)); ?>" title="Editar trabajador/a">✏️</a> |
                        <a href="<?php echo esc_url(add_query_arg('toggle', $trabajador->ID, $base)); ?>"
                           onclick="return confirm('¿Seguro que deseas cambiar el estado de este trabajador?');"
                           title="<?php echo ($trabajador->user_estado === 'Activado') ? 'Desactivar trabajador' : 'Activar trabajador'; ?>">
                           <?php echo ($trabajador->user_estado === 'Activado') ? '⛔' : '✅'; ?>
                        </a> |
                        <a href="<?php echo esc_url(add_query_arg('reenviar', $trabajador->ID, $base)); ?>" title="Recuperar contraseña">🔑</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        <?php endif; ?>
        </tbody>
    </table>

    <?php if (isset($_GET['edit'])):
        $edit_id = intval($_GET['edit']);
        $trabajador = $wpdb->get_row($wpdb->prepare("SELECT * FROM mnkt_users WHERE ID = %d", $edit_id));
        if ($trabajador && $trabajador->user_cif === $empresa_sel->user_cif): ?>
            <h3>Editar Trabajador</h3>
            <form method="post" class="pure-form pure-form-stacked">
                <input type="hidden" name="trabajador_id" value="<?php echo $trabajador->ID; ?>">
                <label>Nombre: <input type="text" name="nombre" value="<?php echo esc_attr($trabajador->display_name); ?>" required></label>
                <label>Email: <input type="email" name="email" value="<?php echo esc_attr($trabajador->user_email); ?>" required></label>
                <label>Nº DNI (sin la letra): <input type="text" name="dni" value="<?php echo esc_attr($trabajador->user_dni); ?>" pattern="[0-9]{6,8}" required></label>
                <input type="submit" name="editar_trabajador" class="pure-button pure-button-primary" value="Actualizar">
            </form>
        <?php endif; ?>
    <?php endif; ?>

    <?php
    return ob_get_clean();
}
