<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Obtener ranking de puntos por empresa y periodo (FRONT).
 */
if (!function_exists('ch_ch_get_ranking_puntos_front')) {
    function ch_ch_get_ranking_puntos_front($user_cif, $fecha_inicio, $fecha_fin) {
        global $wpdb;

        $tabla_registro = $wpdb->prefix . 'registro_horario';
        $tabla_users    = $wpdb->prefix . 'users';

        $sql = $wpdb->prepare("
            SELECT 
                u.ID,
                u.display_name,
                u.user_email,
                u.user_rol,
                COALESCE(SUM(r.puntos), 0) AS total_puntos,
                COUNT(r.id) AS registros
            FROM $tabla_registro r
            JOIN $tabla_users u ON u.ID = r.user_id
            WHERE r.cif_usuario = %s
              AND r.entrada BETWEEN %s AND %s
              AND u.user_estado = 'Activado'
              AND u.user_rol IN ('Trabajador', 'Gestor')
            GROUP BY u.ID
            HAVING total_puntos <> 0
            ORDER BY total_puntos DESC, u.display_name ASC
        ", $user_cif, $fecha_inicio . ' 00:00:00', $fecha_fin . ' 23:59:59');

        $result = $wpdb->get_results($sql);
        return $result ?: [];
    }
}

/**
 * Tabla HTML de ranking para el front.
 */
if (!function_exists('ch_ch_build_ranking_table_html_front')) {
    function ch_ch_build_ranking_table_html_front($ranking) {

        if (empty($ranking)) {
            return '<p style="text-align:center; color:#666;">No hay puntos registrados en este periodo.</p>';
        }

        $html  = '<div style="overflow-x:auto;">';
        $html .= '<table class="pure-table pure-table-striped" style="width:100%; max-width:100%; font-size:14px;">';
        $html .= '<thead>';
        $html .= '<tr>';
        $html .= '<th style="text-align:left;">Pos.</th>';
        $html .= '<th style="text-align:left;">Nombre</th>';
        $html .= '<th style="text-align:center;">Rol</th>';
        $html .= '<th style="text-align:right;">Puntos</th>';
        $html .= '<th style="text-align:right;">Registros</th>';
        $html .= '</tr>';
        $html .= '</thead><tbody>';

        $pos = 1;
        foreach ($ranking as $row) {
            $medalla = '';
            if ($pos === 1)      $medalla = ' 🥇';
            elseif ($pos === 2)  $medalla = ' 🥈';
            elseif ($pos === 3)  $medalla = ' 🥉';

            $rol_label = ($row->user_rol === 'Gestor') ? 'Gestor' : 'Trabajador';

            $html .= '<tr>';
            $html .= '<td>' . esc_html($pos) . $medalla . '</td>';
            $html .= '<td>' . esc_html($row->display_name) . '</td>';
            $html .= '<td style="text-align:center; color:#555;">' . esc_html($rol_label) . '</td>';
            $html .= '<td style="text-align:right; font-weight:600;">' . esc_html($row->total_puntos) . '</td>';
            $html .= '<td style="text-align:right; color:#777;">' . esc_html($row->registros) . '</td>';
            $html .= '</tr>';

            $pos++;
        }

        $html .= '</tbody></table></div>';
        return $html;
    }
}

/**
 * Shortcode FRONT: [ranking_gamificacion]
 * Vista de ranking para el Gestor en el front.
 */
function ch_ch_ranking_gamificacion_front_shortcode() {
    if (!is_user_logged_in()) {
        return '<p class="pure-alert" style="text-align:center;">Debes iniciar sesión para ver el ranking.</p>';
    }

    global $wpdb;
    $current_user_id = get_current_user_id();

    // Comprobar que el usuario es Gestor y está activado
    $user_info = $wpdb->get_row($wpdb->prepare(
        "SELECT user_rol, user_estado FROM {$wpdb->prefix}users WHERE ID = %d",
        $current_user_id
    ));

    if (!$user_info || $user_info->user_rol !== 'Gestor' || $user_info->user_estado !== 'Activado') {
        return '<p class="pure-alert" style="text-align:center;">No tienes permisos para acceder a este ranking.</p>';
    }

    // Obtener empresas asociadas como Gestor
    $empresas = $wpdb->get_results($wpdb->prepare("
        SELECT DISTINCT e.id, e.user_cif, e.razon_social
        FROM {$wpdb->prefix}empresas_usuarios eu
        JOIN {$wpdb->prefix}empresas e ON e.id = eu.empresa_id
        WHERE eu.user_id = %d
          AND eu.rol_en_empresa = 'Gestor'
          AND eu.estado = 'Activado'
        ORDER BY e.razon_social ASC, e.user_cif ASC
    ", $current_user_id));

    if (!$empresas) {
        return '<p class="pure-alert" style="text-align:center;">No tienes ninguna empresa asignada como Gestor.</p>';
    }

    // Valores de formulario (POST)
    $hoy         = current_time('Y-m-d');
    $anio_actual = (int) date('Y');

    $empresa_id   = isset($_POST['rg_empresa_id']) ? intval($_POST['rg_empresa_id']) : (int)$empresas[0]->id;
    $tipo_periodo = isset($_POST['rg_tipo_periodo']) ? sanitize_text_field($_POST['rg_tipo_periodo']) : 'mes';
    $mes          = isset($_POST['rg_mes']) ? intval($_POST['rg_mes']) : intval(date('n'));
    $trimestre    = isset($_POST['rg_trimestre']) ? intval($_POST['rg_trimestre']) : 1;
    $anio         = isset($_POST['rg_anio']) ? intval($_POST['rg_anio']) : $anio_actual;

    $empresa_activa = null;
    foreach ($empresas as $e) {
        if ((int)$e->id === $empresa_id) {
            $empresa_activa = $e;
            break;
        }
    }
    if (!$empresa_activa) {
        $empresa_activa = $empresas[0];
        $empresa_id     = (int)$empresa_activa->id;
    }

    $user_cif       = $empresa_activa->user_cif;
    $empresa_nombre = $empresa_activa->razon_social ?: $empresa_activa->user_cif;

    // Calcular fechas según tipo de periodo
    $fecha_inicio = $hoy;
    $fecha_fin    = $hoy;
    $periodo_texto = '';

    if ($tipo_periodo === 'mes') {
        $fecha_inicio = sprintf('%04d-%02d-01', $anio, $mes);
        $fecha_fin    = date('Y-m-t', strtotime($fecha_inicio));
        $periodo_texto = sprintf('Mes %02d/%04d', $mes, $anio);
    } elseif ($tipo_periodo === 'trimestre') {
        $mes_inicio   = ($trimestre - 1) * 3 + 1;
        $fecha_inicio = sprintf('%04d-%02d-01', $anio, $mes_inicio);
        $mes_fin      = $mes_inicio + 2;
        $fecha_fin    = date('Y-m-t', strtotime(sprintf('%04d-%02d-01', $anio, $mes_fin)));
        $periodo_texto = sprintf('Trimestre %d de %04d', $trimestre, $anio);
    } else { // año
        $fecha_inicio = sprintf('%04d-01-01', $anio);
        $fecha_fin    = sprintf('%04d-12-31', $anio);
        $periodo_texto = sprintf('Año %04d', $anio);
    }

    // Ranking (solo si ya se ha cargado el formulario o hay valores por defecto)
    $ranking            = ch_ch_get_ranking_puntos_front($user_cif, $fecha_inicio, $fecha_fin);
    $tabla_ranking_html = ch_ch_build_ranking_table_html_front($ranking);

    ob_start();
    ?>

    <div class="rh-contenedor" style="max-width:900px; margin:0 auto;">
        <h2 style="text-align:center; margin-bottom:10px;">🏆 Ranking de gamificación</h2>
        <p style="text-align:center; margin-top:0; color:#555;">
            Empresa: <strong><?php echo esc_html($empresa_nombre); ?></strong><br>
            Periodo: <strong><?php echo esc_html($periodo_texto); ?></strong>
        </p>
        <form method="post" class="pure-form pure-form-stacked" style="margin:20px 0; padding:16px 18px; border:1px solid #ddd; border-radius:10px; background:#f9fafb;">
            <div style="display:flex; flex-wrap:wrap; gap:16px; align-items:flex-end;">

                <?php if (count($empresas) > 1): ?>
                    <div style="min-width:220px;">
                        <label for="rg_empresa_id" style="font-weight:600; margin-bottom:4px; display:block;">Empresa</label>
                        <select name="rg_empresa_id" id="rg_empresa_id" style="width:100%; max-width:280px;">
                            <?php foreach ($empresas as $e): 
                                $label = ($e->razon_social ?: $e->user_cif) . ' (' . $e->user_cif . ')';
                            ?>
                                <option value="<?php echo esc_attr($e->id); ?>" <?php selected($e->id, $empresa_id); ?>>
                                    <?php echo esc_html($label); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                <?php else: ?>
                    <input type="hidden" name="rg_empresa_id" value="<?php echo esc_attr($empresa_id); ?>">
                <?php endif; ?>

                <!-- Selector de periodo con “botones” -->
                <div style="flex:1; min-width:260px;">
                    <div style="font-weight:600; margin-bottom:6px;">Periodo</div>

                    <div style="display:flex; gap:8px; flex-wrap:wrap; margin-bottom:10px;">
                        <!-- MES -->
                        <label style="
                            cursor:pointer;
                            padding:6px 12px;
                            border-radius:999px;
                            border:1px solid #ccc;
                            font-size:13px;
                            <?php echo ($tipo_periodo === 'mes') ? 'background:#0073aa;color:#fff;border-color:#0073aa;' : 'background:#fff;color:#333;'; ?>
                        ">
                            <input type="radio" name="rg_tipo_periodo" value="mes" 
                                <?php checked($tipo_periodo, 'mes'); ?>
                                style="display:none;">
                            Mes
                        </label>

                        <!-- TRIMESTRE -->
                        <label style="
                            cursor:pointer;
                            padding:6px 12px;
                            border-radius:999px;
                            border:1px solid #ccc;
                            font-size:13px;
                            <?php echo ($tipo_periodo === 'trimestre') ? 'background:#0073aa;color:#fff;border-color:#0073aa;' : 'background:#fff;color:#333;'; ?>
                        ">
                            <input type="radio" name="rg_tipo_periodo" value="trimestre" 
                                <?php checked($tipo_periodo, 'trimestre'); ?>
                                style="display:none;">
                            Trimestre
                        </label>

                        <!-- AÑO -->
                        <label style="
                            cursor:pointer;
                            padding:6px 12px;
                            border-radius:999px;
                            border:1px solid #ccc;
                            font-size:13px;
                            <?php echo ($tipo_periodo === 'anio') ? 'background:#0073aa;color:#fff;border-color:#0073aa;' : 'background:#fff;color:#333;'; ?>
                        ">
                            <input type="radio" name="rg_tipo_periodo" value="anio" 
                                <?php checked($tipo_periodo, 'anio'); ?>
                                style="display:none;">
                            Año
                        </label>
                    </div>

                    <!-- Fila con selectores de mes / trimestre / año -->
                    <div style="display:flex; gap:10px; flex-wrap:wrap; align-items:flex-end;">

                        <div id="rg_selector_mes" style="<?php echo ($tipo_periodo === 'mes') ? '' : 'display:none;'; ?>">
                            <label for="rg_mes" style="font-size:12px; margin-bottom:2px; display:block;">Mes</label>
                            <select name="rg_mes" id="rg_mes" style="min-width:90px;">
                                <?php for ($m = 1; $m <= 12; $m++): ?>
                                    <option value="<?php echo esc_attr($m); ?>" <?php selected($mes, $m); ?>>
                                        <?php echo esc_html($m); ?>
                                    </option>
                                <?php endfor; ?>
                            </select>
                        </div>

                        <div id="rg_selector_trimestre" style="<?php echo ($tipo_periodo === 'trimestre') ? '' : 'display:none;'; ?>">
                            <label for="rg_trimestre" style="font-size:12px; margin-bottom:2px; display:block;">Trimestre</label>
                            <select name="rg_trimestre" id="rg_trimestre" style="min-width:90px;">
                                <option value="1" <?php selected($trimestre, 1); ?>>1º</option>
                                <option value="2" <?php selected($trimestre, 2); ?>>2º</option>
                                <option value="3" <?php selected($trimestre, 3); ?>>3º</option>
                                <option value="4" <?php selected($trimestre, 4); ?>>4º</option>
                            </select>
                        </div>

                        <div>
                            <label for="rg_anio" style="font-size:12px; margin-bottom:2px; display:block;">Año</label>
                            <input type="number" name="rg_anio" id="rg_anio" 
                                value="<?php echo esc_attr($anio); ?>" 
                                style="width:90px;">
                        </div>

                    </div>
                </div>

                <div>
                    <button type="submit" class="pure-button pure-button-primary" style="margin-top:4px;">
                        🔄 Actualizar
                    </button>
                </div>
            </div>
        </form>

       

        <div style="margin-top:10px;">
            <?php echo $tabla_ranking_html; ?>
        </div>

        <p style="margin-top:10px; font-size:12px; color:#777; text-align:center;">
            Los puntos reflejan el uso correcto del sistema de fichaje (no quién echa más horas).
        </p>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const radios = document.querySelectorAll('input[name="rg_tipo_periodo"]');
        const selMes = document.getElementById('rg_selector_mes');
        const selTri = document.getElementById('rg_selector_trimestre');

        function actualizarVisibilidad() {
            const tipo = document.querySelector('input[name="rg_tipo_periodo"]:checked').value;
            if (tipo === 'mes') {
                selMes.style.display = '';
                selTri.style.display = 'none';
            } else if (tipo === 'trimestre') {
                selMes.style.display = 'none';
                selTri.style.display = '';
            } else {
                selMes.style.display = 'none';
                selTri.style.display = 'none';
            }
        }

        radios.forEach(r => r.addEventListener('change', actualizarVisibilidad));
        actualizarVisibilidad();
    });
    </script>

    <?php
    return ob_get_clean();
}
add_shortcode('ranking_gamificacion', 'ch_ch_ranking_gamificacion_front_shortcode');
