<?php

if (!defined('ABSPATH')) exit;

function lectorqr_enqueue_scripts() {
    wp_enqueue_script('html5-qrcode', 'https://unpkg.com/html5-qrcode', [], null, true);
    wp_enqueue_style('purecss', 'https://cdn.jsdelivr.net/npm/purecss@2.0.6/build/pure-min.css', [], '2.0.6');
}
add_action('wp_enqueue_scripts', 'lectorqr_enqueue_scripts');

function lectorqr_shortcode() {
    ob_start();
    ?>
    <div class="container-plugin">
        <h3>Lector de Código QR para Control Horario</h3>
        <form id="lectorqr-form" class="pure-form pure-form-stacked">
            <button type="button" id="btn-iniciar-lector" class="pure-button pure-button-primary">Iniciar lector QR</button>
            <button type="button" id="btn-detener-lector" class="pure-button pure-button-danger" style="display: none;">Detener lector</button>

            <div id="lector-qr" style="width: 300px; height: 300px; margin: 20px auto; display: none;"></div>

            <label for="latitud">Latitud:</label>
            <input type="text" id="latitud" readonly required>

            <label for="longitud">Longitud:</label>
            <input type="text" id="longitud" readonly required>
        </form>

        <p id="mensaje"></p>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function () {
        const btnIniciar = document.getElementById('btn-iniciar-lector');
        const btnDetener = document.getElementById('btn-detener-lector');
        const contenedorLector = document.getElementById('lector-qr');
        const mensaje = document.getElementById('mensaje');
        let html5QrCode = null;

        // Pre-cargar geolocalización
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(function(position) {
                document.getElementById('latitud').value = position.coords.latitude.toFixed(8);
                document.getElementById('longitud').value = position.coords.longitude.toFixed(8);
            });
        }

        btnIniciar.addEventListener('click', () => {
            contenedorLector.style.display = 'block';
            btnDetener.style.display = 'inline-block';
            btnIniciar.disabled = true;

            html5QrCode = new Html5Qrcode("lector-qr");
            html5QrCode.start(
                { facingMode: "environment" }, // trasera por defecto
                { fps: 10, qrbox: 250 },
                qrCodeMessage => {
                    html5QrCode.stop().then(() => {
                        contenedorLector.style.display = 'none';
                        btnDetener.style.display = 'none';
                        btnIniciar.disabled = false;
                        validarQR(qrCodeMessage.trim());
                    });
                },
                error => console.warn('QR no detectado:', error)
            );
        });

        btnDetener.addEventListener('click', () => {
            if (html5QrCode) {
                html5QrCode.stop().then(() => {
                    contenedorLector.style.display = 'none';
                    btnDetener.style.display = 'none';
                    btnIniciar.disabled = false;
                });
            }
        });

        function validarQR(dni) {
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=lectorqr_validar&dni=${encodeURIComponent(dni)}`
            })
            .then(res => res.json())
            .then(data => {
                mensaje.innerHTML = data.mensaje;
                if (data.registro_permitido) {
                    // Ya NO mandamos 'tipo'; el servidor decide
                    registrarQR(dni);
                }
            })
            .catch(() => {
                mensaje.innerHTML = "<p style='color:red'>Error validando el QR.</p>";
            });
        }

        function registrarQR(dni) {
            const latitud = document.getElementById("latitud").value;
            const longitud = document.getElementById("longitud").value;

            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=lectorqr_registrar&dni=${encodeURIComponent(dni)}&latitud=${encodeURIComponent(latitud)}&longitud=${encodeURIComponent(longitud)}`
            })
            .then(res => res.json())
            .then(data => {
                mensaje.innerHTML = data.mensaje || 'Registro completado.';
                setTimeout(() => location.reload(), 3000);
            })
            .catch(() => {
                mensaje.innerHTML = "<p style='color:red'>Error registrando el fichaje.</p>";
            });
        }
    });
    </script>
    <?php
    return ob_get_clean();
}
add_shortcode('lector-qr', 'lectorqr_shortcode');

/** AJAX validar: comprueba token y muestra mensaje (el tipo aquí es solo informativo) */
add_action('wp_ajax_lectorqr_validar', 'lectorqr_validar');
add_action('wp_ajax_nopriv_lectorqr_validar', 'lectorqr_validar');
function lectorqr_validar() {
    global $wpdb;
    $token = isset($_POST['dni']) ? sanitize_text_field($_POST['dni']) : '';

    $usuario = $wpdb->get_row($wpdb->prepare(
        "SELECT ID, user_cif FROM {$wpdb->prefix}users WHERE qr_token = %s",
        $token
    ));

    if ($usuario) {
        // Informativo: si la última línea está abierta → siguiente es salida
        $tipo = $wpdb->get_var($wpdb->prepare(
            "SELECT salida IS NULL FROM {$wpdb->prefix}registro_horario 
             WHERE user_id = %d AND cif_usuario = %s 
             ORDER BY entradaR DESC 
             LIMIT 1",
            $usuario->ID, $usuario->user_cif
        )) ? 'salida' : 'entrada';

        wp_send_json([
            'registro_permitido' => true,
            'tipo' => $tipo,
            'mensaje' => "<p style='color: green;'>Token válido. Procesando...</p>"
        ]);
    } else {
        wp_send_json([
            'registro_permitido' => false,
            'mensaje' => "<p style='color: red;'>Token no válido.</p>"
        ]);
    }
}

/** AJAX registrar: decisión autoritativa en servidor (evita dobles entradas) */
add_action('wp_ajax_lectorqr_registrar', 'lectorqr_registrar');
add_action('wp_ajax_nopriv_lectorqr_registrar', 'lectorqr_registrar');
function lectorqr_registrar() {
    global $wpdb;

    $token = isset($_POST['dni']) ? sanitize_text_field($_POST['dni']) : '';
    $lat   = isset($_POST['latitud'])  ? floatval($_POST['latitud'])  : null;
    $lng   = isset($_POST['longitud']) ? floatval($_POST['longitud']) : null;
    $ts    = current_time('mysql'); // hora local WP

    $usuario = $wpdb->get_row($wpdb->prepare(
        "SELECT ID, user_cif FROM {$wpdb->prefix}users WHERE qr_token = %s",
        $token
    ));

    if (!$usuario) {
        wp_send_json(['mensaje' => '<p style="color: red;">Token no válido.</p>']);
    }

    $nombre = get_user_by('ID', $usuario->ID)->display_name;

    // ¿Hay una entrada abierta?
    $pendiente = $wpdb->get_row($wpdb->prepare(
        "SELECT id, entrada FROM {$wpdb->prefix}registro_horario 
         WHERE user_id = %d AND cif_usuario = %s AND salida IS NULL 
         ORDER BY entradaR DESC 
         LIMIT 1",
        $usuario->ID, $usuario->user_cif
    ));

    if (!$pendiente) {
        // ENTRADA
        $wpdb->insert("{$wpdb->prefix}registro_horario", [
            'user_id'          => $usuario->ID,
            'cif_usuario'      => $usuario->user_cif,
            'entrada'          => $ts,
            'entradaR'         => $ts,
            'latitud_entrada'  => $lat,
            'longitud_entrada' => $lng
        ]);

        // Rotamos token para el siguiente fichaje (tu QR en web debe refrescarse)
        $nuevo_token = bin2hex(random_bytes(16));
        $wpdb->update("{$wpdb->prefix}users", ['qr_token' => $nuevo_token], ['ID' => $usuario->ID]);

        wp_send_json([
            'mensaje' => "<p style='color: green;'>Hola <strong>$nombre</strong>, has registrado tu <strong>entrada</strong>.</p>"
        ]);
    } else {
        // SALIDA
        $tiempo_total = $wpdb->get_var($wpdb->prepare(
            "SELECT TIMEDIFF(%s, entrada) FROM {$wpdb->prefix}registro_horario WHERE id = %d",
            $ts, $pendiente->id
        ));

        $wpdb->update("{$wpdb->prefix}registro_horario", [
            'salida'           => $ts,
            'salidaR'          => $ts,
            'latitud_salida'   => $lat,
            'longitud_salida'  => $lng,
            'tiempo_total'     => $tiempo_total,
            'tiempo_totalR'    => $tiempo_total
        ], ['id' => $pendiente->id]);

        do_action('registrar_salida_trabajador', $usuario->ID, $pendiente->id);

        // Rotamos token para el siguiente fichaje
        $nuevo_token = bin2hex(random_bytes(16));
        $wpdb->update("{$wpdb->prefix}users", ['qr_token' => $nuevo_token], ['ID' => $usuario->ID]);

        wp_send_json([
            'mensaje' => "<p style='color: green;'>Hola <strong>$nombre</strong>, has registrado tu <strong>salida</strong>.</p>"
        ]);
    }
}