<?php
if (!defined('ABSPATH')) exit;

global $wpdb;
$tabla        = $wpdb->prefix . 'empresas';
$tabla_users  = $wpdb->prefix . 'users';
$tabla_m2m    = $wpdb->prefix . 'empresas_usuarios';

$empresa_a_editar = null;
$mensaje = '';
$resultado = null;

// ===== Helpers =====

// Inserta (idempotente) una vinculación gestor<->empresa.
// Si la tabla M2M tiene 'rol_en_empresa' y 'estado', fuerza 'Gestor'/'Activado'.
function eg_ensure_m2m_link_company($tabla_m2m, $user_id, $empresa_id) {
    global $wpdb;
    if ($user_id <= 0 || $empresa_id <= 0) return false;

    $has_rol    = (bool) $wpdb->get_var( $wpdb->prepare("SHOW COLUMNS FROM $tabla_m2m LIKE %s", 'rol_en_empresa') );
    $has_estado = (bool) $wpdb->get_var( $wpdb->prepare("SHOW COLUMNS FROM $tabla_m2m LIKE %s", 'estado') );

    $cols = ['user_id','empresa_id','creado_en'];
    $vals = ['%d','%d','%s'];
    $data = [$user_id, $empresa_id, current_time('mysql')];

    if ($has_rol)   { $cols[]='rol_en_empresa'; $vals[]='%s'; $data[]='Gestor'; }
    if ($has_estado){ $cols[]='estado';         $vals[]='%s'; $data[]='Activado'; }

    $cols_sql     = implode(',', $cols);
    $placeholders = implode(',', $vals);

    $sql = $wpdb->prepare(
        "INSERT INTO $tabla_m2m ($cols_sql) VALUES ($placeholders)
         ON DUPLICATE KEY UPDATE empresa_id = empresa_id",
        ...$data
    );
    $res = $wpdb->query($sql);
    if ($res !== false) return true;

    // Fallback si no hay UNIQUE:
    $where = "user_id=%d AND empresa_id=%d";
    $args  = [$user_id, $empresa_id];
    if ($has_rol) { $where .= " AND rol_en_empresa=%s"; $args[]='Gestor'; }
    $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $tabla_m2m WHERE $where LIMIT 1", ...$args));
    if ($exists) return true;

    return (bool) $wpdb->insert(
        $tabla_m2m,
        array_combine($cols, $data),
        $vals
    );
}

// ===== Cargar empresa a editar (si aplica) =====
if (isset($_GET['editar']) && is_numeric($_GET['editar'])) {
    $empresa_a_editar = $wpdb->get_row(
        $wpdb->prepare("SELECT * FROM $tabla WHERE id = %d", intval($_GET['editar']))
    );
}

// ===== Acciones POST (crear/actualizar/desvincular/vincular gestores) =====
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Crear/Actualizar empresa
    if (
        isset($_POST['crear_empresa']) &&
        isset($_POST['crear_empresa_nonce_field']) &&
        wp_verify_nonce($_POST['crear_empresa_nonce_field'], 'crear_empresa_nonce')
    ) {
        $empresa_id = isset($_POST['empresa_id']) ? intval($_POST['empresa_id']) : 0;

        $datos = [
            'user_cif'               => sanitize_text_field($_POST['user_cif']),
            'razon_social'           => sanitize_text_field($_POST['razon_social']),
            'direccion_facturacion'  => sanitize_textarea_field($_POST['direccion_facturacion'] ?? ''),
            'estado_empresa'         => sanitize_text_field($_POST['estado_empresa']),
            'qr_general'             => sanitize_text_field($_POST['qr_general']),
            'estado_lugares'         => sanitize_text_field($_POST['estado_lugares']),
            'metodo_pago'            => sanitize_text_field($_POST['metodo_pago']),
            'gamificacion'           => sanitize_text_field($_POST['gamificacion']),
            'estado_envios_trabajadores' => sanitize_text_field($_POST['estado_envios_trabajadores'] ?? 'activado'),

        ];

        if ($empresa_id > 0) {
            // Actualizar
            $resultado = $wpdb->update($tabla, $datos, ['id' => $empresa_id]);
            if ($resultado !== false) {
                $mensaje = '<div class="notice notice-success is-dismissible"><p>✅ Empresa actualizada correctamente.</p></div>';
                // Refresca objeto empresa para el panel de gestores
                $empresa_a_editar = $wpdb->get_row($wpdb->prepare("SELECT * FROM $tabla WHERE id=%d", $empresa_id));
            } else {
                $mensaje = '<div class="notice notice-error"><p>❌ Error al actualizar: ' . esc_html($wpdb->last_error) . '</p></div>';
            }
        } else {
            // Crear
            $datos_extra = [
                'user_id'                    => get_current_user_id(),
                'estado_validacion'          => 'Pendiente',
                'aceptacion_proteccion_datos'=> 0,
                'aceptacion_sepa'            => 0,
                'fecha_aceptacion'           => current_time('mysql'), // creación
                'documentos_generados'       => 0,
                'total_lugares'              => 0,
            ];
            $resultado = $wpdb->insert($tabla, array_merge($datos, $datos_extra));
            if ($resultado !== false) {
                $new_id = (int)$wpdb->insert_id;
                $empresa_a_editar = $wpdb->get_row($wpdb->prepare("SELECT * FROM $tabla WHERE id=%d", $new_id));
                $mensaje = '<div class="notice notice-success is-dismissible"><p>✅ Empresa creada correctamente. Ahora puedes asignar gestores abajo.</p></div>';
            } else {
                $mensaje = '<div class="notice notice-error"><p>❌ Error al guardar la empresa: ' . esc_html($wpdb->last_error) . '</p></div>';
            }
        }
    }

    // Vincular gestor a empresa (solo si ya tenemos empresa_id)
    if (
        isset($_POST['asignar_gestor']) &&
        isset($_POST['asignar_gestor_nonce']) &&
        wp_verify_nonce($_POST['asignar_gestor_nonce'], 'asignar_gestor') &&
        !empty($_POST['empresa_id_gest']) && !empty($_POST['gestor_id'])
    ) {
        $empresa_id_gest = intval($_POST['empresa_id_gest']);
        $gestor_id       = intval($_POST['gestor_id']);

        $ok = eg_ensure_m2m_link_company($tabla_m2m, $gestor_id, $empresa_id_gest);
        if ($ok) {
            $mensaje .= '<div class="notice notice-success is-dismissible"><p>✅ Gestor asignado.</p></div>';
        } else {
            $mensaje .= '<div class="notice notice-error"><p>❌ No se pudo asignar el gestor.</p></div>';
        }

        // Mantener empresa cargada
        $empresa_a_editar = $wpdb->get_row($wpdb->prepare("SELECT * FROM $tabla WHERE id=%d", $empresa_id_gest));
    }

    // Desvincular gestor de empresa (eliminar relación)
    if (
        isset($_POST['desvincular_gestor']) &&
        isset($_POST['desvincular_gestor_nonce']) &&
        wp_verify_nonce($_POST['desvincular_gestor_nonce'], 'desvincular_gestor') &&
        !empty($_POST['rel_id']) && !empty($_POST['empresa_id_gest'])
    ) {
        $rel_id          = intval($_POST['rel_id']);
        $empresa_id_gest = intval($_POST['empresa_id_gest']);
        $del = $wpdb->delete($tabla_m2m, ['id' => $rel_id], ['%d']);
        if ($del) {
            $mensaje .= '<div class="notice notice-success is-dismissible"><p>🗑️ Vinculación eliminada.</p></div>';
        } else {
            $mensaje .= '<div class="notice notice-error"><p>❌ No se pudo eliminar la vinculación.</p></div>';
        }
        $empresa_a_editar = $wpdb->get_row($wpdb->prepare("SELECT * FROM $tabla WHERE id=%d", $empresa_id_gest));
    }
}

// ===== Filtro CIF (listado) =====
$filtro_cif = isset($_GET['filtro_cif']) ? sanitize_text_field($_GET['filtro_cif']) : '';
$consulta = "SELECT * FROM $tabla";
if ($filtro_cif) {
    $consulta .= $wpdb->prepare(" WHERE user_cif LIKE %s", "%$filtro_cif%");
}
$consulta .= " ORDER BY id DESC";
$empresas = $wpdb->get_results($consulta);

// ===== Datos para el panel de asignación (solo si hay empresa cargada) =====
$gestores_disponibles = [];
$gestores_vinculados  = [];
$q_gestor = isset($_GET['q_gestor']) ? sanitize_text_field($_GET['q_gestor']) : '';

if ($empresa_a_editar && !empty($empresa_a_editar->id)) {
    $empresa_id_actual = (int)$empresa_a_editar->id;

    // Gestores ya vinculados (con rel_id para eliminar)
    $gestores_vinculados = $wpdb->get_results($wpdb->prepare("
        SELECT eu.id AS rel_id, u.ID, u.display_name, u.user_email
        FROM $tabla_m2m eu
        JOIN $tabla_users u ON u.ID = eu.user_id
        WHERE eu.empresa_id = %d
          AND u.user_rol = 'Gestor'
        ORDER BY u.display_name ASC
    ", $empresa_id_actual));

    // IDs ya vinculados
    $ids_vinc = array_map(fn($r)=> (int)$r->ID, $gestores_vinculados);
    $ids_vinc = $ids_vinc ? implode(',', array_map('intval', $ids_vinc)) : '0';

    // Gestores disponibles (no vinculados) con búsqueda opcional
    $where = "WHERE u.user_rol='Gestor' AND u.user_estado='Activado'";
    $args  = [];
    if ($q_gestor !== '') {
        $where .= " AND (u.display_name LIKE %s OR u.user_email LIKE %s)";
        $args[] = "%$q_gestor%";
        $args[] = "%$q_gestor%";
    }
    $gestores_disponibles = $wpdb->get_results(
        $wpdb->prepare("
            SELECT u.ID, u.display_name, u.user_email
            FROM $tabla_users u
            $where
            AND u.ID NOT IN ($ids_vinc)
            ORDER BY u.display_name ASC
            LIMIT 300
        ", ...$args)
    );
}

// ===== Estilos =====
?>
<style>
/* Layout básico bonito */
.wrap .card {
  background:#fff; border:1px solid #e5e5e5; border-radius:8px; padding:16px; margin:14px 0;
}
.grid-2 { display:grid; grid-template-columns: 1fr 1fr; gap:16px; }
.grid-3 { display:grid; grid-template-columns: 1fr 1fr 1fr; gap:16px; }
@media (max-width: 900px) { .grid-2, .grid-3 { grid-template-columns: 1fr; } }

.label { font-weight:600; margin-bottom:6px; display:block; }
.input, .select, .textarea { width:100%; }
.textarea { min-height: 90px; }

/* Toggles (estilo iOS) */
.switch { position:relative; display:inline-block; width:50px; height:26px; vertical-align:middle; margin:0 8px; }
.switch input { opacity:0; width:0; height:0; }
.slider-toggle { position:absolute; inset:0; background:#ccc; transition:.2s; border-radius:999px; cursor:pointer; }
.slider-toggle:before { content:""; position:absolute; height:22px; width:22px; left:2px; top:2px; background:#fff; transition:.2s; border-radius:50%; box-shadow:0 1px 3px rgba(0,0,0,.25); }
.switch input:checked + .slider-toggle { background:#2ea44f; }
.switch input:checked + .slider-toggle:before { transform:translateX(24px); }
.toggle-label { font-weight:600; margin-right:8px; }
.pill { display:inline-block; padding:2px 8px; border-radius:999px; font-size:12px; }
.pill.on { background:#e6ffed; color:#0a7; }
.pill.off{ background:#ffe9e9; color:#c00; }

/* Pills de gestores */
.gestor-pill {
  display:inline-block; padding:2px 6px; margin:2px 4px 2px 0;
  background:#f1f1f1; border-radius:4px; font-family:monospace; font-size:12px; color:#333;
  border:1px solid #ddd;
}
.small-muted { color:#666; font-size:12px; }
</style>

<div class="wrap">
    <h1 class="wp-heading-inline">🏢 Gestión de Empresas</h1>
    <?php echo $mensaje; ?>

    <div class="card">
        <h2 style="margin-top:0;"><?php echo $empresa_a_editar ? '✏️ Editar empresa' : '➕ Crear nueva empresa'; ?></h2>
        <form method="post">
            <?php wp_nonce_field('crear_empresa_nonce', 'crear_empresa_nonce_field'); ?>
            <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_a_editar->id ?? ''); ?>">

            <div class="grid-2">
                <div>
                    <label class="label" for="user_cif">CIF</label>
                    <input name="user_cif" id="user_cif" type="text" class="input" required value="<?php echo esc_attr($empresa_a_editar->user_cif ?? ''); ?>">
                </div>
                <div>
                    <label class="label" for="razon_social">Razón Social</label>
                    <input name="razon_social" id="razon_social" type="text" class="input" required value="<?php echo esc_attr($empresa_a_editar->razon_social ?? ''); ?>">
                </div>
            </div>

            <div class="grid-2">
                <div>
                    <label class="label" for="metodo_pago">Método de Pago</label>
                    <select name="metodo_pago" id="metodo_pago" class="select">
                        <?php
                        $mp = $empresa_a_editar->metodo_pago ?? '';
                        $opts = ['' => '—', 'Domiciliación' => 'Domiciliación', 'Tarjeta' => 'Tarjeta', 'Transferencia' => 'Transferencia'];
                        foreach ($opts as $v => $lbl) {
                            echo '<option value="'.esc_attr($v).'" '.selected($mp, $v, false).'>'.$lbl.'</option>';
                        }
                        ?>
                    </select>
                </div>
               
            </div>
 <br><br>
            <div class="grid-3" style="align-items:center;">
                <div>
                    <span class="toggle-label">Empresa</span>
                    <label class="switch">
                      <input type="checkbox" id="cb_estado_empresa" <?php checked(($empresa_a_editar->estado_empresa ?? 'activado'), 'activado'); ?>>
                      <span class="slider-toggle"></span>
                    </label>
                    <span id="pill_estado_empresa" class="pill"></span>
                    <input type="hidden" name="estado_empresa" id="estado_empresa" value="<?php echo esc_attr($empresa_a_editar->estado_empresa ?? 'activado'); ?>">
                </div>

                <div>
                    <span class="toggle-label">QR general</span>
                    <label class="switch">
                      <input type="checkbox" id="cb_qr_general" <?php checked(($empresa_a_editar->qr_general ?? 'desactivado'), 'activado'); ?>>
                      <span class="slider-toggle"></span>
                    </label>
                    <span id="pill_qr_general" class="pill"></span>
                    <input type="hidden" name="qr_general" id="qr_general" value="<?php echo esc_attr($empresa_a_editar->qr_general ?? 'desactivado'); ?>">
                </div>

                <div>
                    <span class="toggle-label">Lugares</span>
                    <label class="switch">
                      <input type="checkbox" id="cb_estado_lugares" <?php checked(($empresa_a_editar->estado_lugares ?? 'desactivado'), 'activado'); ?>>
                      <span class="slider-toggle"></span>
                    </label>
                    <span id="pill_estado_lugares" class="pill"></span>
                    <input type="hidden" name="estado_lugares" id="estado_lugares" value="<?php echo esc_attr($empresa_a_editar->estado_lugares ?? 'desactivado'); ?>">
                </div>
            </div>

            <div style="margin-top:10px;">
                <span class="toggle-label">Gamificación</span>
                <label class="switch">
                  <input type="checkbox" id="cb_gamificacion" <?php checked(($empresa_a_editar->gamificacion ?? 'desactivado'), 'activado'); ?>>
                  <span class="slider-toggle"></span>
                </label>
                <span id="pill_gamificacion" class="pill"></span>
                <input type="hidden" name="gamificacion" id="gamificacion" value="<?php echo esc_attr($empresa_a_editar->gamificacion ?? 'desactivado'); ?>">
            </div>

            <div style="margin-top:10px;">
                <span class="toggle-label">Env. trabajadores</span>
                <label class="switch">
                <input type="checkbox" id="cb_envios_trabajadores" <?php checked(($empresa_a_editar->estado_envios_trabajadores ?? 'activado'), 'activado'); ?>>
                <span class="slider-toggle"></span>
                </label>
                <span id="pill_envios_trabajadores" class="pill"></span>
                <input type="hidden" name="estado_envios_trabajadores" id="estado_envios_trabajadores" value="<?php echo esc_attr($empresa_a_editar->estado_envios_trabajadores ?? 'activado'); ?>">
            </div>


            <p class="submit" style="margin-top:18px;">
                <button type="submit" name="crear_empresa" class="button button-primary">
                    <?php echo $empresa_a_editar ? 'Actualizar empresa' : 'Guardar empresa'; ?>
                </button>
            </p>
        </form>
    </div>

    <?php if ($empresa_a_editar && !empty($empresa_a_editar->id)) : ?>
    <div class="card">
        <h2 style="margin-top:0;">👔 Gestores de esta empresa</h2>
        <p class="small-muted">Asigna gestores ya creados. Puedes buscar por nombre o email.</p>

        <!-- Buscador de gestores no vinculados -->
        <form method="get" style="display:flex; gap:10px; align-items:flex-end; flex-wrap:wrap; margin-bottom:10px;">
            <?php
            foreach ($_GET as $k=>$v) {
                if (in_array($k, ['q_gestor'], true)) continue;
                echo '<input type="hidden" name="'.esc_attr($k).'" value="'.esc_attr($v).'">';
            }
            // Garantizar parámetro 'editar' para permanecer en la misma empresa
            if (!isset($_GET['editar'])) {
                echo '<input type="hidden" name="editar" value="'.esc_attr($empresa_a_editar->id).'">';
            }
            ?>
            <div>
                <label class="label">Buscar gestor</label>
                <input type="text" name="q_gestor" value="<?php echo esc_attr($q_gestor); ?>" placeholder="Nombre o email" class="input" style="min-width:280px;">
            </div>
            <div>
                <button class="button">Buscar</button>
            </div>
        </form>

        <!-- Asignar uno -->
        <form method="post" style="display:flex; gap:10px; align-items:flex-end; flex-wrap:wrap;">
            <?php wp_nonce_field('asignar_gestor', 'asignar_gestor_nonce'); ?>
            <input type="hidden" name="empresa_id_gest" value="<?php echo esc_attr($empresa_a_editar->id); ?>">
            <div style="min-width:360px; max-width:520px; flex:1;">
                <label class="label">Seleccionar gestor (activado, no vinculado)</label>
                <select name="gestor_id" class="select">
                    <?php if (empty($gestores_disponibles)): ?>
                        <option value="">— No hay gestores disponibles con este filtro —</option>
                    <?php else: ?>
                        <option value="">— Selecciona gestor —</option>
                        <?php foreach ($gestores_disponibles as $g): ?>
                            <option value="<?php echo esc_attr($g->ID); ?>">
                                <?php echo esc_html($g->display_name ?: "Usuario #{$g->ID}"); ?> — <?php echo esc_html($g->user_email); ?>
                            </option>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </select>
            </div>
            <div>
                <button type="submit" name="asignar_gestor" class="button button-primary">Asignar gestor</button>
            </div>
        </form>

        <hr>

        <!-- Gestores ya vinculados -->
        <h3 style="margin:10px 0 8px;">Vinculados ahora</h3>
        <?php if (empty($gestores_vinculados)): ?>
            <div class="notice notice-info"><p>No hay gestores vinculados todavía.</p></div>
        <?php else: ?>
            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Gestor</th>
                        <th>Email</th>
                        <th style="width:120px;">Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($gestores_vinculados as $g): ?>
                        <tr>
                            <td><span class="gestor-pill" title="<?php echo esc_attr(($g->display_name ?: '—') . ' — ' . ($g->user_email ?: '—')); ?>"><?php echo esc_html($g->ID); ?></span></td>
                            <td><?php echo esc_html($g->display_name ?: "Usuario #{$g->ID}"); ?></td>
                            <td><?php echo esc_html($g->user_email); ?></td>
                            <td>
                                <form method="post" onsubmit="return confirm('¿Desvincular este gestor?');" style="display:inline;">
                                    <?php wp_nonce_field('desvincular_gestor', 'desvincular_gestor_nonce'); ?>
                                    <input type="hidden" name="rel_id" value="<?php echo esc_attr($g->rel_id); ?>">
                                    <input type="hidden" name="empresa_id_gest" value="<?php echo esc_attr($empresa_a_editar->id); ?>">
                                    <button type="submit" name="desvincular_gestor" class="button button-small">Desvincular</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    <?php endif; ?>

   

    <div class="card">
        <h2 style="margin-top:0;">📋 Empresas registradas</h2>
        <?php if ($empresas): ?>
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>CIF</th>
                        <th>Razón social</th>
                        <th>Método de pago</th>
                        <th>Fecha creación</th>
                        <th>Estado</th>
                        <th>QR</th>
                        <th>Lugares</th>
                        <th>Gamificación</th>
                        <th>Env. trabajadores</th>
                        <th>Gestores</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($empresas as $empresa): ?>
                    <?php
                    // Gestores vinculados (solo Gestores; estado lo mostramos todos, si quieres solo activados añade AND u.user_estado='Activado')
                    $gestores_list = $wpdb->get_results($wpdb->prepare("
                        SELECT u.ID, u.display_name, u.user_email
                        FROM $tabla_m2m eu
                        JOIN $tabla_users u ON u.ID = eu.user_id
                        WHERE eu.empresa_id = %d AND u.user_rol='Gestor'
                        ORDER BY u.display_name ASC
                    ", $empresa->id));

                    $gestores_html = '—';
                    if (!empty($gestores_list)) {
                        $tmp = [];
                        foreach ($gestores_list as $g) {
                            $tmp[] = '<span class="gestor-pill" title="'.esc_attr(($g->display_name ?: '—').' — '.($g->user_email ?: '—')).'">'.esc_html($g->ID).'</span>';
                        }
                        $gestores_html = implode(' ', $tmp);
                    }

                    // Pills de estado
                    $pill = function($valor, $ok='activado'){ $on = strtolower((string)$valor)===strtolower($ok); return '<span class="pill '.($on?'on':'off').'">'.($on?'Activado':'Desactivado').'</span>'; };
                    ?>
                    <tr>
                        <td><?php echo esc_html($empresa->id); ?></td>
                        <td><?php echo esc_html($empresa->user_cif); ?></td>
                        <td style="max-width: 280px;"><?php echo esc_html($empresa->razon_social); ?></td>
                        <td><?php echo esc_html($empresa->metodo_pago ?: '—'); ?></td>
                        <td><?php echo esc_html($empresa->fecha_aceptacion ?: '—'); ?></td>
                        <td><?php echo $pill($empresa->estado_empresa, 'activado'); ?></td>
                        <td><?php echo $pill($empresa->qr_general, 'activado'); ?></td>
                        <td><?php echo $pill($empresa->estado_lugares, 'activado'); ?></td>
                        <td><?php echo $pill($empresa->gamificacion ?? 'desactivado', 'activado'); ?></td>
                        <td><?php echo $pill($empresa->estado_envios_trabajadores ?? 'activado', 'activado'); ?></td>

                        <td style="max-width:320px;"><?php echo $gestores_html; ?></td>
                        <td>
                            <a href="?page=<?php echo esc_attr($_GET['page']); ?>&tab=empresas&editar=<?php echo esc_attr($empresa->id); ?>" class="button">✏️ Editar</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>No hay empresas registradas.</p>
        <?php endif; ?>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', () => {
  const map = [
    {cb:'#cb_estado_empresa', hidden:'#estado_empresa', pill:'#pill_estado_empresa'},
    {cb:'#cb_qr_general', hidden:'#qr_general', pill:'#pill_qr_general'},
    {cb:'#cb_estado_lugares', hidden:'#estado_lugares', pill:'#pill_estado_lugares'},
    {cb:'#cb_gamificacion', hidden:'#gamificacion', pill:'#pill_gamificacion'},
    {cb:'#cb_envios_trabajadores', hidden:'#estado_envios_trabajadores', pill:'#pill_envios_trabajadores'},

  ];
  function setState(row){
    const cb = document.querySelector(row.cb);
    if(!cb) return;
    const hidden = document.querySelector(row.hidden);
    const pill = document.querySelector(row.pill);
    const checked = cb.checked;
    hidden.value = checked ? 'activado' : 'desactivado';
    if (pill) {
      pill.textContent = checked ? 'Activado' : 'Desactivado';
      pill.className = 'pill ' + (checked ? 'on' : 'off');
    }
  }
  map.forEach(row => {
    setState(row);
    const cb = document.querySelector(row.cb);
    if (cb) cb.addEventListener('change', () => setState(row));
  });
});
</script>
