<?php
/**
 * Módulo Admin: Gestión de Suscripciones Stripe
 * - Menú "Control Horario" + submenú "Suscripciones Stripe"
 * - Tabla con empresas, filtros y acciones
 * - Ajuste de clave secreta de Stripe (opción en BD)
 */
if (!defined('ABSPATH')) exit;

if (!class_exists('CH_Stripe_Admin_Subscriptions')) {

class CH_Stripe_Admin_Subscriptions {

    const OPT_SECRET_KEY = 'ch_stripe_secret_key';

    /** cache rápido de columnas por tabla */
private $table_columns_cache = [];

private function table_has_column($table, $column) {
    global $wpdb;
    $cache_key = $table . '::' . $column;
    if (isset($this->table_columns_cache[$cache_key])) return $this->table_columns_cache[$cache_key];
    $exists = (bool) $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = %s AND COLUMN_NAME = %s",
        $table, $column
    ));
    $this->table_columns_cache[$cache_key] = $exists;
    return $exists;
}

private function get_cached_pm_status($empresa_id) {
    $opt = get_option('ch_pm_' . intval($empresa_id));
    if (is_array($opt)) return $opt; // ['has'=>0|1,'ts'=>'Y-m-d H:i:s']
    return null;
}

private function set_cached_pm_status($empresa_id, $has) {
    update_option('ch_pm_' . intval($empresa_id), [
        'has' => $has ? 1 : 0,
        'ts'  => current_time('mysql'),
    ], false);
}


    public function __construct() {
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_post_ch_stripe_save_key', [$this, 'handle_save_key']);
        add_action('admin_post_ch_stripe_action',   [$this, 'handle_row_action']);
    }

    public function register_admin_menu() {
    // Menú raíz (apunta al Dashboard)
    add_menu_page(
        'Control Horario',
        'Control Horario',
        'manage_options',
        'ch-control-horario',             // slug raíz
        'ch_render_ch_stripe_dashboard',  // callback del dashboard (función global en dashboard.php)
        'dashicons-clock',
        58
    );

    // Submenú: Dashboard (misma página que el raíz)
    add_submenu_page(
        'ch-control-horario',
        'Dashboard CH Stripe',
        'Dashboard',
        'manage_options',
        'ch-control-horario',             // mismo slug que el raíz
        'ch_render_ch_stripe_dashboard'
    );

    // Submenú: Suscripciones
    add_submenu_page(
        'ch-control-horario',
        'Suscripciones Stripe',
        'Suscripciones',
        'manage_options',
        'ch-suscripciones-stripe',
        [$this, 'render_admin_page']
    );
}


    public function render_dashboard_stub() {
        echo '<div class="wrap"><h1>Control Horario</h1><p>Panel principal del plugin.</p></div>';
    }

    public function handle_save_key() {
        if (!current_user_can('manage_options')) wp_die('No permitido');
        check_admin_referer('ch_stripe_save_key');

        $key = isset($_POST['stripe_secret_key']) ? sanitize_text_field($_POST['stripe_secret_key']) : '';
        update_option(self::OPT_SECRET_KEY, $key);
        wp_redirect(add_query_arg(['page'=>'ch-suscripciones-stripe', 'updated'=>'1'], admin_url('admin.php')));
        exit;
    }

public function handle_row_action() {
    if (!current_user_can('manage_options')) wp_die('No permitido');
    check_admin_referer('ch_stripe_row_action');

    global $wpdb;
    $t = 'mnkt_empresas';

    $empresa_id = isset($_POST['empresa_id']) ? intval($_POST['empresa_id']) : 0;
    $do         = isset($_POST['do']) ? sanitize_key($_POST['do']) : '';

    if (!$do) {
        wp_redirect(add_query_arg(['page'=>'ch-suscripciones-stripe', 'err'=>'1'], admin_url('admin.php')));
        exit;
    }

    switch ($do) {
        case 'mark_converted':
            if ($empresa_id) {
                $wpdb->update($t, [
                    'trial_status'   => 'convertido',
                    'estado_empresa' => 'activado',
                ], ['id'=>$empresa_id]);
            }
            break;

        case 'deactivate':
            if ($empresa_id) {
                $wpdb->update($t, [
                    'estado_empresa' => 'desactivado',
                    'deactivated_at' => current_time('mysql'),
                ], ['id'=>$empresa_id]);
            }
            break;

        case 'activate':
            if ($empresa_id) {
                $wpdb->update($t, [
                    'estado_empresa' => 'activado',
                ], ['id'=>$empresa_id]);
            }
            break;

        case 'refresh_from_stripe':
            if ($empresa_id) {
                $this->refresh_from_stripe($empresa_id);
            }
            break;

        case 'refresh_all':
            @ignore_user_abort(true);
            @set_time_limit(300);
            $ids = $wpdb->get_col("
                SELECT id FROM {$t}
                WHERE (stripe_customer_id IS NOT NULL AND stripe_customer_id <> '')
                   OR (stripe_subscription_id IS NOT NULL AND stripe_subscription_id <> '')
            ");
            $count = 0;
            if ($ids) {
                foreach ($ids as $eid) {
                    $this->refresh_from_stripe((int)$eid);
                    $count++;
                }
            }
            wp_redirect(add_query_arg([
                'page' => 'ch-suscripciones-stripe',
                'done' => 'refresh_all',
                'n'    => $count
            ], admin_url('admin.php')));
            exit;

       case 'send_pm_reminder':
    if ($empresa_id) {
        $res = $this->send_pm_reminder($empresa_id);

        if (is_wp_error($res)) {
            wp_redirect(add_query_arg([
                'page' => isset($_POST['from_page']) ? sanitize_key($_POST['from_page']) : 'ch-recordatorios-pm',
                'done' => 'send_pm_reminder',
                'ok'   => 0,
                'err'  => $res->get_error_code(),
                'msg'  => rawurlencode($res->get_error_message()),
            ], admin_url('admin.php')));
        } else {
            wp_redirect(add_query_arg([
                'page' => isset($_POST['from_page']) ? sanitize_text_field($_POST['from_page']) : 'ch-suscripciones-stripe',
                'done'      => 'send_pm_reminder',
                'ok'        => 1,
                'email'     => rawurlencode($res['email']),
                'empresa'   => rawurlencode($res['empresa']),
                'cif'       => rawurlencode($res['cif']),
                'days_left' => (int)$res['days_left'],
                'trial_end' => rawurlencode($res['trial_end']),
                'subject'   => rawurlencode($res['subject']),
            ], admin_url('admin.php')));
        }
        exit;
    }
    break;



        case 'send_pm_reminder_all':
            $n = $this->send_pm_reminder_all();
            wp_redirect(add_query_arg([
                'page' => 'ch-recordatorios-pm',
                'done' => 'send_pm_reminder_all',
                'n'    => intval($n)
            ], admin_url('admin.php')));
            exit;

        default:
            break;
    }

    wp_redirect(add_query_arg(['page'=>'ch-suscripciones-stripe', 'done'=>$do], admin_url('admin.php')));
    exit;
}


   private function refresh_from_stripe($empresa_id) {
    global $wpdb;
    $t = 'mnkt_empresas';
    $row = $wpdb->get_row($wpdb->prepare(
        "SELECT id, stripe_subscription_id, stripe_customer_id FROM {$t} WHERE id=%d",
        $empresa_id
    ));
    if (!$row) return;

    $secret = get_option(self::OPT_SECRET_KEY, '');

    
    if (!$secret) return;

    // Cargar SDK (por si aún no está)
    $sdk = CH_STRIPE_WH_PLUGIN_DIR . 'lib/stripe-php/init.php';
    if (file_exists($sdk)) require_once $sdk;
    if (!class_exists('\Stripe\StripeClient')) return;

    $has_pm = false; // <- lo que queremos determinar

    try {
        $client = new \Stripe\StripeClient($secret);

        // 1) Si hay subscription id, refrescamos estado
        if (!empty($row->stripe_subscription_id)) {
            $sub = $client->subscriptions->retrieve($row->stripe_subscription_id, []);
            if ($sub && !empty($sub->status) && $sub->status === 'active') {
                $wpdb->update($t, [
                    'trial_status'           => 'convertido',
                    'estado_empresa'         => 'activado',
                    'stripe_subscription_id' => $sub->id,
                ], ['id'=>$empresa_id]);
            }
        }

        // 2) Comprobar método de pago guardado
        if (!empty($row->stripe_customer_id)) {
            // a) mira default_payment_method en invoice_settings
            $cus = $client->customers->retrieve($row->stripe_customer_id, []);
            if ($cus && !empty($cus->invoice_settings) && !empty($cus->invoice_settings->default_payment_method)) {
                $has_pm = true;
            }

            // b) si aún no, lista PMs tipo card (1 basta)
            if (!$has_pm) {
                $methods = $client->paymentMethods->all([
                    'customer' => $row->stripe_customer_id,
                    'type'     => 'card',
                    'limit'    => 1,
                ]);
                $has_pm = !empty($methods->data);
            }
        }

        // 3) Persistimos el estado: si existen columnas, a la tabla; si no, a options.
        $has_col = $this->table_has_column($t, 'has_payment_method');
        if ($has_col) {
            $set = [
                'has_payment_method'         => $has_pm ? 1 : 0,
                'payment_method_last_checked'=> current_time('mysql'),
            ];
            $wpdb->update($t, $set, ['id'=>$empresa_id]);
        } else {
            $this->set_cached_pm_status($empresa_id, $has_pm);
        }

    } catch (\Throwable $e) {
        // log “suave”: no rompas la UI
        if (method_exists($this, 'log')) {
            $this->log('Stripe refresh ERROR', ['e'=>$e->getMessage(), 'empresa_id'=>$empresa_id]);
        }
    }
}

/** ======== Recordatorios PM: helpers ======== */

private function pm_opt_key_sent($empresa_id, $days_left){
    return 'ch_pm_reminder_' . intval($empresa_id) . '_' . intval($days_left);
}

private function already_sent_today($empresa_id, $days_left){
    $key = $this->pm_opt_key_sent($empresa_id, $days_left);
    $val = get_option($key);
    if (!$val) return false;
    $last = strtotime($val);
    return (date_i18n('Y-m-d', $last) === date_i18n('Y-m-d', current_time('timestamp')));
}

private function mark_sent_today($empresa_id, $days_left){
    $key = $this->pm_opt_key_sent($empresa_id, $days_left);
    update_option($key, current_time('mysql'), false);
}

private function empresa_has_pm_cached_or_db($e){
    // Usa columna si existe; si no, option cache usada en admin
    $has_col = $this->table_has_column('mnkt_empresas', 'has_payment_method');
    if ($has_col && isset($e->has_payment_method)) {
        return (int)$e->has_payment_method === 1;
    }
    $cache = get_option('ch_pm_' . intval($e->id));
    if (is_array($cache) && array_key_exists('has', $cache)) {
        return (int)$cache['has'] === 1;
    }
    return null; // desconocido
}

private function compute_days_left_local($trial_expires_at){
    if (!$trial_expires_at) return null;
    $expires_local = strtotime(get_date_from_gmt($trial_expires_at));
    if (!$expires_local) return null;
    $now_ts = current_time('timestamp');
    return (int) ceil( ($expires_local - $now_ts) / DAY_IN_SECONDS );
}

private function send_pm_reminder($empresa_id){
    global $wpdb;
    $t_empresas = 'mnkt_empresas';
    $t_rel      = 'mnkt_empresas_usuarios';
    $t_users    = 'mnkt_users';

    $e = $wpdb->get_row($wpdb->prepare("
        SELECT e.id, e.user_cif, e.razon_social, e.trial_status, e.trial_expires_at,
               e.stripe_customer_id, e.has_payment_method
        FROM {$t_empresas} e
        WHERE e.id = %d
    ", $empresa_id));
    if (!$e) return new \WP_Error('no_empresa', 'Empresa no encontrada');

    if ($e->trial_status !== 'en_curso')
        return new \WP_Error('no_trial', 'La empresa no está en trial');

    if (empty($e->trial_expires_at))
        return new \WP_Error('no_trial_end', 'La fecha de fin de trial está vacía');

    $days_left = $this->compute_days_left_local($e->trial_expires_at);
    if ($days_left === null)
        return new \WP_Error('no_days_left', 'No se han podido calcular los días restantes');

    $has_pm = $this->empresa_has_pm_cached_or_db($e);
    if ($has_pm === true)
        return new \WP_Error('has_pm', 'La empresa ya tiene método de pago guardado');

    $gestor = $wpdb->get_row($wpdb->prepare("
        SELECT u.user_email, COALESCE(NULLIF(u.display_name,''), u.user_login) AS name, u.user_login
        FROM {$t_rel} r
        INNER JOIN {$t_users} u ON u.ID = r.user_id
        WHERE r.empresa_id = %d AND r.rol_en_empresa='Gestor' AND r.estado='Activado'
        ORDER BY r.creado_en ASC
        LIMIT 1
    ", $empresa_id));
    if (!$gestor || empty($gestor->user_email))
        return new \WP_Error('no_gestor', 'No hay gestor activo con email');

    $base = defined('CH_STRIPE_WH_PLUGIN_DIR') ? CH_STRIPE_WH_PLUGIN_DIR : trailingslashit(dirname(__DIR__));
    $tpl  = trailingslashit($base . 'templates-stripe') . 'reminder-no-pm.html';

    if (file_exists($tpl)) {
        $html = file_get_contents($tpl);
    } else {
        $html = '
            <h2>Recordatorio de método de pago</h2>
            <p>Hola {{name}}, todavía no tenemos un método de pago guardado para {{empresa}} ({{cif}}).</p>
            <p>Quedan <strong>{{days_left}}</strong> día(s). El periodo de prueba termina el <strong>{{trial_end}}</strong>.</p>
            <p><a href="{{portal_link}}">Añadir/actualizar método de pago</a></p>
        ';
    }

    $portal_link = add_query_arg('ch_stripe_portal', '1', site_url('/'));
    $trial_end_h = date_i18n('d/m/Y H:i', strtotime(get_date_from_gmt($e->trial_expires_at)));

    $repl = [
        '{{name}}'        => esc_html($gestor->name ?: $gestor->user_login),
        '{{empresa}}'     => esc_html($e->razon_social ?: ''),
        '{{cif}}'         => esc_html($e->user_cif),
        '{{days_left}}'   => esc_html($days_left),
        '{{trial_end}}'   => esc_html($trial_end_h),
        '{{portal_link}}' => esc_url($portal_link),
    ];
    $body = strtr($html, $repl);

    $headers = [
        'Content-Type: text/html; charset=UTF-8',
        'From: Despapela <no-reply@despapela.com>',
        'Reply-To: Soporte <soporte@despapela.com>',
    ];
    $subject = 'Recordatorio: añade tu método de pago';

    $sent = wp_mail($gestor->user_email, $subject, $body, $headers);

    error_log("CH_STRIPE: PM-REMINDER empresa={$empresa_id} email={$gestor->user_email} sent=" . ($sent ? '1' : '0'));

    if (!$sent) {
        return new \WP_Error('mail_failed', 'wp_mail devolvió false');
    }

    $this->mark_sent_today($empresa_id, $days_left);

    return [
        'ok'        => 1,
        'email'     => $gestor->user_email,
        'empresa'   => $e->razon_social ?: '',
        'cif'       => $e->user_cif,
        'days_left' => (int)$days_left,
        'trial_end' => $trial_end_h,
        'subject'   => $subject,
    ];
}




private function send_pm_reminder_all(){
    global $wpdb;
    $t_empresas = 'mnkt_empresas';
    $rows = $wpdb->get_results("
        SELECT e.id, e.trial_status, e.trial_expires_at, e.has_payment_method
        FROM {$t_empresas} e
        WHERE e.trial_status='en_curso'
          AND e.trial_expires_at IS NOT NULL
          AND (e.has_payment_method IS NULL OR e.has_payment_method=0)
        ORDER BY e.trial_expires_at ASC
    ");
    $n=0;
    if ($rows) {
        foreach ($rows as $e) {
            if ($this->send_pm_reminder((int)$e->id)) $n++;
        }
    }
    return $n;
}


public function render_admin_page() {
    if (!current_user_can('manage_options')) wp_die('No permitido');

    global $wpdb;
    $t_empresas = 'mnkt_empresas';
    $t_rel      = 'mnkt_empresas_usuarios';
    $t_users    = 'mnkt_users';

    // ¿La tabla tiene la columna has_payment_method?
    $has_pm_column = $this->table_has_column($t_empresas, 'has_payment_method');

    // Filtros
    $q    = isset($_GET['s']) ? trim(sanitize_text_field($_GET['s'])) : '';
    $page = max(1, isset($_GET['paged']) ? intval($_GET['paged']) : 1);
    $ppp  = 20;
    $off  = ($page - 1) * $ppp;

    // Query base con join para email gestor (si existe)
$where = "e.metodo_pago = 'Tarjeta'";
    $params = [];
    if ($q !== '') {
        $where .= " AND (e.user_cif LIKE %s OR e.razon_social LIKE %s OR u.user_email LIKE %s)";
        $like = '%' . $wpdb->esc_like($q) . '%';
        $params[] = $like; $params[] = $like; $params[] = $like;
    }

    // Añade columnas extras si existen en BD
    $extra_cols = '';
    if ($has_pm_column) {
        $extra_cols = ", e.has_payment_method, e.payment_method_last_checked";
    }

    $sql = "
        SELECT SQL_CALC_FOUND_ROWS
            e.id, e.user_cif, e.razon_social, e.estado_empresa, e.trial_status,
            e.trial_started_at, e.trial_expires_at, e.metodo_pago,
            e.stripe_customer_id, e.stripe_subscription_id,
            u.user_email AS gestor_email
            {$extra_cols}
        FROM {$t_empresas} e
        LEFT JOIN {$t_rel} r ON r.empresa_id = e.id AND r.rol_en_empresa='Gestor'
        LEFT JOIN {$t_users} u ON u.ID = r.user_id
        WHERE {$where}
        GROUP BY e.id
        ORDER BY e.id DESC
        LIMIT %d OFFSET %d
    ";

    $rows  = $wpdb->get_results($wpdb->prepare($sql, array_merge($params, [$ppp, $off])));
    $total = (int)$wpdb->get_var("SELECT FOUND_ROWS()");

    $secret = get_option(self::OPT_SECRET_KEY, '');

    ?>
    <div class="wrap">
        <h1>Suscripciones Stripe</h1>

        <?php if (isset($_GET['updated'])): ?>
            <div class="notice notice-success"><p>Configuración guardada.</p></div>
        <?php endif; ?>
     <?php if (isset($_GET['done'])): ?>
  <div class="notice notice-success">
    <p>
      Acción ejecutada: <code><?php echo esc_html($_GET['done']); ?></code>
      <?php if (!empty($_GET['n'])): ?>
        — Refrescadas: <strong><?php echo intval($_GET['n']); ?></strong>
      <?php elseif (isset($_GET['ok'])): ?>
        — Enviado: <strong><?php echo intval($_GET['ok']); ?></strong>
      <?php endif; ?>
    </p>
  </div>
<?php endif; ?>


<form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin: 12px 0;">
    <?php wp_nonce_field('ch_stripe_row_action'); ?>
    <input type="hidden" name="action" value="ch_stripe_action">
    <input type="hidden" name="do" value="refresh_all">
    
    <?php if (empty($secret)): ?>
        <span style="color:#777;margin-left:8px;">Configura la clave secreta de Stripe para habilitarlo.</span>
    <?php endif; ?>
</form>

        <h2 class="title">Ajustes de Stripe</h2>
       <div style="display:flex;gap:12px;align-items:flex-end;flex-wrap:wrap;margin:12px 0 16px 0;">

  <!-- Form: guardar clave -->
  <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin:0;">
    <?php wp_nonce_field('ch_stripe_save_key'); ?>
    <input type="hidden" name="action" value="ch_stripe_save_key">

    <table class="form-table" role="presentation" style="margin:0;">
      <tr>
        <th scope="row" style="padding-bottom:0;">
          <label for="stripe_secret_key">Clave secreta Stripe</label>
        </th>
        <td style="padding-bottom:0;">
          <input id="stripe_secret_key" type="password" name="stripe_secret_key"
                 value="<?php echo esc_attr($secret); ?>" class="regular-text"
                 placeholder="sk_live_... / sk_test_...">
          <p class="description" style="margin-top:6px;">
            Se guarda en la base de datos. Necesaria para “Refrescar desde Stripe”.
          </p>
        </td>
      </tr>
    </table>

    <?php submit_button('Guardar clave', 'primary', 'submit', /*wrap*/ false); ?>
  </form>

  <!-- Form: refrescar todo -->
  <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin:0;display:flex;align-items:center;gap:8px;">
    <?php wp_nonce_field('ch_stripe_row_action'); ?>
    <input type="hidden" name="action" value="ch_stripe_action">
    <input type="hidden" name="do" value="refresh_all">

    <button class="button button-secondary"
            <?php disabled(empty($secret)); ?>
            onclick="return confirm('Esto actualizará TODAS las empresas con datos en Stripe. ¿Continuar?');">
      Refrescar todo
    </button>

    <?php if (empty($secret)): ?>
      <span style="color:#777;">Configura la clave secreta de Stripe para habilitarlo.</span>
    <?php endif; ?>
  </form>

</div>



        <hr/>

        <form method="get" style="margin: 12px 0;">
            <input type="hidden" name="page" value="ch-suscripciones-stripe">
            <p class="search-box">
                <label class="screen-reader-text" for="empresa-search-input">Buscar:</label>
                <input type="search" id="empresa-search-input" name="s" value="<?php echo esc_attr($q); ?>">
                <input type="submit" id="search-submit" class="button" value="Buscar">
            </p>
        </form>

        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>CIF</th>
                    <th>Razón social</th>
                    <th>Gestor (email)</th>
                    <th>Estado</th>
                    <th>Trial</th>
                    <th>Stripe</th>
                    <th>Método guardado</th> <!-- NUEVA COLUMNA -->
                    <th>Acciones</th>
                </tr>
            </thead>
            <tbody>
            <?php if ($rows): foreach ($rows as $r): ?>
                <tr>
                    <td><?php echo intval($r->id); ?></td>
                    <td><code><?php echo esc_html($r->user_cif); ?></code></td>
                    <td><?php echo esc_html($r->razon_social ?: '—'); ?></td>
                    <td><?php echo esc_html($r->gestor_email ?: '—'); ?></td>
                    <td><?php echo esc_html($r->estado_empresa); ?></td>
                    <td>
                        <?php
                            echo 'Estado: ' . esc_html($r->trial_status ?: '—') . '<br/>';
                            if ($r->trial_started_at) {
                                echo 'Inicio: ' . esc_html( date_i18n('d/m/Y', strtotime($r->trial_started_at)) ) . '<br/>';
                            }
                            if ($r->trial_expires_at) {
                                echo 'Fin: ' . esc_html( date_i18n('d/m/Y', strtotime(get_date_from_gmt($r->trial_expires_at))) );
                            }
                        ?>
                    </td>
                    <td style="font-size:12px;">
                        <?php
                            echo 'Customer: ' . ($r->stripe_customer_id ? '<code>'.esc_html($r->stripe_customer_id).'</code>' : '—') . '<br/>';
                            echo 'Sub: ' . ($r->stripe_subscription_id ? '<code>'.esc_html($r->stripe_subscription_id).'</code>' : '—');
                        ?>
                    </td>
                    <td>
                        <?php
                            if ($has_pm_column && isset($r->has_payment_method)) {
                                echo $r->has_payment_method ? 'Sí' : 'No';
                                if (!empty($r->payment_method_last_checked)) {
                                    echo '<br><span style="color:#777;font-size:11px;">' .
                                        esc_html( date_i18n('d/m/y H:i', strtotime($r->payment_method_last_checked)) ) .
                                    '</span>';
                                }
                            } else {
                                // sin columna en BD: lee caché de options (tras pulsar “Refrescar”)
                                $cached = $this->get_cached_pm_status($r->id);
                                if ($cached) {
                                    echo $cached['has'] ? 'Sí' : 'No';
                                    echo '<br><span style="color:#777;font-size:11px;">' .
                                        esc_html( date_i18n('d/m/y H:i', strtotime($cached['ts'])) ) .
                                    '</span>';
                                } else {
                                    echo '—';
                                }
                            }
                        ?>
                    </td>
                    <td>
                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                            <?php wp_nonce_field('ch_stripe_row_action'); ?>
                            <input type="hidden" name="action" value="ch_stripe_action">
                            <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                            <input type="hidden" name="do" value="refresh_from_stripe">
                            <button class="button" <?php disabled(empty($secret)); ?>>Refrescar</button>
                        </form>

                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                            <?php wp_nonce_field('ch_stripe_row_action'); ?>
                            <input type="hidden" name="action" value="ch_stripe_action">
                            <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                            <input type="hidden" name="do" value="mark_converted">
                            <button class="button">Marcar convertido</button>
                        </form>

                        <?php if ($r->estado_empresa === 'activado'): ?>
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                                <?php wp_nonce_field('ch_stripe_row_action'); ?>
                                <input type="hidden" name="action" value="ch_stripe_action">
                                <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                                <input type="hidden" name="do" value="deactivate">
                                <button class="button button-secondary">Desactivar</button>
                            </form>
                        <?php else: ?>
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                                <?php wp_nonce_field('ch_stripe_row_action'); ?>
                                <input type="hidden" name="action" value="ch_stripe_action">
                                <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                                <input type="hidden" name="do" value="activate">
                                <button class="button button-primary">Reactivar</button>
                            </form>

                            

                        <?php endif; ?>
                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                                <?php wp_nonce_field('ch_stripe_row_action'); ?>
                                <input type="hidden" name="action" value="ch_stripe_action">
                                <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                                <input type="hidden" name="do" value="send_pm_reminder">
                                <input type="hidden" name="from_page" value="ch-suscripciones-stripe">
                                <button class="button">Recordatorio PM</button>
                            </form>
                    </td>
                </tr>
            <?php endforeach; else: ?>
                <tr><td colspan="9">No hay resultados.</td></tr>
            <?php endif; ?>
            </tbody>
        </table>

        <?php
            // Paginación simple
            $total_pages = max(1, ceil($total / $ppp));
            if ($total_pages > 1):
        ?>
            <div class="tablenav">
                <div class="tablenav-pages">
                    <?php
                        $base = add_query_arg(['page'=>'ch-suscripciones-stripe','s'=>$q, 'paged'=>'%#%'], admin_url('admin.php'));
                        echo paginate_links([
                            'base'      => $base,
                            'format'    => '',
                            'current'   => $page,
                            'total'     => $total_pages,
                            'prev_text' => '&laquo;',
                            'next_text' => '&raquo;',
                        ]);
                    ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
    <?php
}

public function render_pm_reminders_page() {
    if (!current_user_can('manage_options')) wp_die('No permitido');
    global $wpdb;

    $t_empresas = 'mnkt_empresas';
    $t_rel      = 'mnkt_empresas_usuarios';
    $t_users    = 'mnkt_users';

    // Búsqueda simple por CIF/razón/email
    $q = isset($_GET['s']) ? trim(sanitize_text_field($_GET['s'])) : '';

    $where = "e.trial_status='en_curso' AND e.trial_expires_at IS NOT NULL";
    $has_pm_col = $this->table_has_column($t_empresas, 'has_payment_method');
    if ($has_pm_col) {
        $where .= " AND (e.has_payment_method IS NULL OR e.has_payment_method=0)";
    }

    $params = [];
    if ($q !== '') {
        $where .= " AND (e.user_cif LIKE %s OR e.razon_social LIKE %s OR u.user_email LIKE %s)";
        $like = '%' . $wpdb->esc_like($q) . '%';
        $params[] = $like; $params[] = $like; $params[] = $like;
    }

    $sql = "
        SELECT e.id, e.user_cif, e.razon_social, e.trial_expires_at, e.trial_status,
               " . ($has_pm_col ? "e.has_payment_method" : "NULL AS has_payment_method") . ",
               u.user_email AS gestor_email,
               COALESCE(NULLIF(u.display_name,''), u.user_login) AS gestor_name
        FROM {$t_empresas} e
        LEFT JOIN {$t_rel} r ON r.empresa_id = e.id AND r.rol_en_empresa='Gestor' AND r.estado='Activado'
        LEFT JOIN {$t_users} u ON u.ID = r.user_id
        WHERE {$where}
        GROUP BY e.id
        ORDER BY e.trial_expires_at ASC
        LIMIT 200
    ";

    $rows = $params ? $wpdb->get_results($wpdb->prepare($sql, $params)) : $wpdb->get_results($sql);

    ?>
    <div class="wrap">
        <h1>Recordatorios Método de Pago</h1>

      <?php if (isset($_GET['done'])): ?>
  <div class="notice notice-success">
    <p style="margin-bottom:6px;">
      Acción ejecutada: <code><?php echo esc_html($_GET['done']); ?></code>
      <?php if (isset($_GET['ok'])): ?>
        — Enviado: <strong><?php echo intval($_GET['ok']); ?></strong>
      <?php endif; ?>
      <?php if (!empty($_GET['n'])): ?>
        — Refrescadas: <strong><?php echo intval($_GET['n']); ?></strong>
      <?php endif; ?>
    </p>

    <?php if (!empty($_GET['ok']) && $_GET['ok'] == 1): ?>
      <p style="margin:4px 0 0;color:#374151;">
        <strong>Destino:</strong> <?php echo esc_html(urldecode($_GET['email'] ?? '')); ?><br/>
        <strong>Empresa:</strong> <?php echo esc_html(urldecode($_GET['empresa'] ?? '')); ?> (CIF: <?php echo esc_html(urldecode($_GET['cif'] ?? '')); ?>)<br/>
        <strong>Días restantes:</strong> <?php echo esc_html($_GET['days_left'] ?? ''); ?> &nbsp;|&nbsp;
        <strong>Fin de trial:</strong> <?php echo esc_html(urldecode($_GET['trial_end'] ?? '')); ?><br/>
        <strong>Asunto:</strong> <?php echo esc_html(urldecode($_GET['subject'] ?? '')); ?>
      </p>
    <?php elseif (isset($_GET['ok']) && intval($_GET['ok']) === 0): ?>
      <p style="margin:4px 0 0;color:#b91c1c;">
        Error: <?php echo esc_html($_GET['err'] ?? ''); ?> — <?php echo esc_html(urldecode($_GET['msg'] ?? '')); ?>
      </p>
    <?php endif; ?>
  </div>
<?php endif; ?>


        <form method="get" style="margin: 12px 0;">
            <input type="hidden" name="page" value="ch-recordatorios-pm">
            <p class="search-box">
                <label class="screen-reader-text" for="rec-search-input">Buscar:</label>
                <input type="search" id="rec-search-input" name="s" value="<?php echo esc_attr($q); ?>">
                <input type="submit" id="search-submit" class="button" value="Buscar">
            </p>
        </form>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin: 0 0 12px 0;">
            <?php wp_nonce_field('ch_stripe_row_action'); ?>
            <input type="hidden" name="action" value="ch_stripe_action">
            <input type="hidden" name="do" value="send_pm_reminder_all">
            <button class="button button-primary"
                onclick="return confirm('Enviar recordatorio a todos los listados?');">
                Enviar recordatorio a todos (listados)
            </button>
        </form>

        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>CIF</th>
                    <th>Razón social</th>
                    <th>Gestor</th>
                    <th>Trial fin</th>
                    <th>Días restantes</th>
                    <th>PM guardado</th>
                    <th>Acción</th>
                </tr>
            </thead>
            <tbody>
            <?php if ($rows): foreach ($rows as $r):
                $days_left = $this->compute_days_left_local($r->trial_expires_at);
                $pm_txt    = ($r->has_payment_method === null) ? '—' : ( (int)$r->has_payment_method ? 'Sí' : 'No' );
            ?>
                <tr>
                    <td><?php echo intval($r->id); ?></td>
                    <td><code><?php echo esc_html($r->user_cif); ?></code></td>
                    <td><?php echo esc_html($r->razon_social ?: '—'); ?></td>
                    <td>
                        <?php
                          echo esc_html($r->gestor_name ?: '—');
                          echo $r->gestor_email ? '<br><code>'.esc_html($r->gestor_email).'</code>' : '';
                        ?>
                    </td>
                    <td><?php echo esc_html( date_i18n('d/m/Y H:i', strtotime(get_date_from_gmt($r->trial_expires_at))) ); ?></td>
                    <td><?php echo ($days_left !== null) ? intval($days_left) : '—'; ?></td>
                    <td><?php echo esc_html($pm_txt); ?></td>
                    <td>
                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                            <?php wp_nonce_field('ch_stripe_row_action'); ?>
                            <input type="hidden" name="action" value="ch_stripe_action">
                            <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                            <input type="hidden" name="do" value="send_pm_reminder">
                            <input type="hidden" name="from_page" value="ch-recordatorios-pm">
                            <button class="button">Enviar recordatorio</button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; else: ?>
                <tr><td colspan="8">No hay empresas pendientes de método de pago (o no hay trials en curso).</td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    <?php
}

   
}

// Instanciar módulo admin
new CH_Stripe_Admin_Subscriptions();

} // end guard class_exists

add_action('wp_mail_failed', function($wp_error){
    error_log('WP_MAIL_FAILED: ' . print_r($wp_error->get_error_message(), true));
});
